import { editorSchema } from './editor-schema.js';
import { logger } from '../engine/logger.js';
import { STORAGE_KEY } from '../constants.js';
import { createCommandId, createDefaultState } from '../default-state.js';

function getDefaultActionForScope(scope) {
  const scopeActions = editorSchema.actions[scope] || [];
  const firstAction = scopeActions[0];
  if (!firstAction) return { id: 'activate', inputs: {} };

  const inputs = {};
  if (firstAction.inputs) {
    for (const input of firstAction.inputs) {
      if (input.default !== undefined) {
        inputs[input.key] = input.default;
      } else if (input.type === 'select' && input.options?.length > 0) {
        inputs[input.key] = input.options[0];
      } else if (input.type === 'checkboxes') {
        inputs[input.key] = [];
      } else {
        inputs[input.key] = '';
      }
    }
  }
  return { id: firstAction.id, inputs };
}

function normalizeCommandAction(action, scope) {
  const scopeActions = editorSchema.actions[scope] || [];
  const rawId = typeof action?.id === 'string' ? action.id : '';
  if (rawId && scopeActions.some((candidate) => candidate.id === rawId)) {
    return { id: rawId, inputs: action?.inputs || {} };
  }

  // Keep unknown IDs as-is so we do not silently rewrite command behavior.
  // Invalid actions can then be surfaced explicitly in the UI.
  if (rawId) {
    return { id: rawId, inputs: action?.inputs || {} };
  }

  return null;
}

function normalizeCommandModel(command) {
  if (!command || typeof command !== 'object') return command;
  const scope = command?.items?.scope || 'tab';
  const normalizedActions = (Array.isArray(command.actions) ? command.actions : [])
    .map((action) => normalizeCommandAction(action, scope))
    .filter(Boolean);

  command.actions = normalizedActions.length > 0
    ? normalizedActions
    : [getDefaultActionForScope(scope)];

  return command;
}

function normalizeLoadedState(loaded) {
  if (!loaded || typeof loaded !== 'object' || !Array.isArray(loaded.commands)) {
    return null;
  }

  const commands = loaded.commands
    .filter((cmd) => cmd && typeof cmd === 'object' && typeof cmd.id === 'string')
    .map((cmd) => normalizeCommandModel(cmd));

  if (commands.length === 0) {
    return null;
  }

  const editingCommandId = commands.some((cmd) => cmd.id === loaded.editingCommandId)
    ? loaded.editingCommandId
    : commands[0].id;

  return {
    ...loaded,
    commands,
    helpPanelOpen: typeof loaded.helpPanelOpen === 'boolean' ? loaded.helpPanelOpen : true,
    editingCommandId
  };
}

function createReactiveState(initialState, onChange) {
  const handler = {
    get(target, key, receiver) {
      const value = Reflect.get(target, key, receiver);
      if (value && typeof value === 'object' && value !== null) {
        return new Proxy(value, handler);
      }
      return value;
    },
    set(target, key, value, receiver) {
      const result = Reflect.set(target, key, value, receiver);
      onChange();
      return result;
    },
    deleteProperty(target, key) {
      const result = Reflect.deleteProperty(target, key);
      onChange();
      return result;
    }
  };
  return new Proxy(initialState, handler);
}

let state = null;
let undoStack = [];
let redoStack = [];
const subscribers = new Set();

function notify() {
  for (const fn of subscribers) {
    fn(state);
  }
}

function pushUndo() {
  undoStack.push(JSON.stringify(state));
  redoStack = [];
}

async function saveToStorage() {
  // Serialize to plain object to ensure Proxy is properly converted for storage
  const plainState = JSON.parse(JSON.stringify(state));
  await chrome.storage.local.set({ [STORAGE_KEY]: plainState });
  chrome.runtime.sendMessage({ type: 'UPDATE_TRIGGERS' });
}

let saveTimer = null;
function debouncedSave() {
  if (saveTimer) clearTimeout(saveTimer);
  saveTimer = setTimeout(() => {
    saveToStorage();
  }, 0);
}

function getEditingCommand() {
  return state.commands.find(c => c.id === state.editingCommandId);
}

export const store = {
  async init() {
    let shouldPersistInitialState = false;

    try {
      const result = await chrome.storage.local.get(STORAGE_KEY);
      const loaded = result[STORAGE_KEY];
      const normalized = normalizeLoadedState(loaded);
      if (normalized) {
        state = createReactiveState(normalized, debouncedSave);
      } else {
        state = createReactiveState(createDefaultState(), debouncedSave);
        shouldPersistInitialState = true;
      }
    } catch (e) {
      logger.warn('Store failed to load state, resetting to default:', e);
      await chrome.storage.local.remove(STORAGE_KEY);
      state = createReactiveState(createDefaultState(), debouncedSave);
      shouldPersistInitialState = true;
    }

    if (shouldPersistInitialState) {
      await chrome.storage.local.set({ [STORAGE_KEY]: JSON.parse(JSON.stringify(state)) });
    }

    if (!state.editingCommandId && state.commands.length > 0) {
      state.editingCommandId = state.commands[0].id;
    }

    return state;
  },

  subscribe(fn) {
    subscribers.add(fn);
    return () => subscribers.delete(fn);
  },

  getState() {
    return state;
  },
  getUndoCount() {
    return undoStack.length;
  },
  getRedoCount() {
    return redoStack.length;
  },

  dispatch(action) {
    logger.log('Store dispatch:', action.type, action);
    
    const undoableActions = [
      'ADD_COMMAND', 'DELETE_COMMAND', 'MOVE_COMMAND', 
      'ADD_FILTER', 'DELETE_FILTER', 'MOVE_FILTER',
      'ADD_ACTION', 'DELETE_ACTION', 'MOVE_ACTION',
      'UPDATE_COMMAND_SCOPE', 'UPDATE_EDITING_COMMAND',
      'UPDATE_EDITING_COMMAND_ITEMS', 'UPDATE_EDITING_COMMAND_REFINEMENT',
      'UPDATE_FILTER', 'UPDATE_ACTION'
    ];
    
    if (undoableActions.includes(action.type)) {
      pushUndo();
    }

    const cmd = getEditingCommand();

    switch (action.type) {
      case 'RESET_STORAGE':
        chrome.storage.local.remove(STORAGE_KEY, () => {
          window.location.reload();
        });
        break;
      case 'ADD_COMMAND': {
        const id = createCommandId();
        state.commands.push({
          id,
          name: '',
          triggers: { icon: false, shortcutSlot: null, context: false, omnibox: null },
          items: { scope: 'tab', selection: 'Active Tab', includeActiveTab: true },
          filters: [],
          refinement: { type: 'All', n: 1 },
          actions: [{ id: 'activate', inputs: {} }],
          executionMode: 'silent'
        });
        state.editingCommandId = id;
        break;
      }
      case 'DELETE_COMMAND':
        state.commands = state.commands.filter(c => c.id !== action.id);
        if (state.editingCommandId === action.id) {
          state.editingCommandId = state.commands[0]?.id || null;
        }
        break;
      case 'SELECT_COMMAND':
        state.editingCommandId = action.id;
        break;
      case 'MOVE_COMMAND': {
        const index = state.commands.findIndex(c => c.id === action.id);
        const newIndex = index + action.direction;
        if (newIndex >= 0 && newIndex < state.commands.length) {
          const [removed] = state.commands.splice(index, 1);
          state.commands.splice(newIndex, 0, removed);
        }
        break;
      }
      case 'UPDATE_COMMAND_SCOPE':
        if (cmd) {
          cmd.items.scope = action.scope;
          const scopeItems = editorSchema.items[action.scope];
          cmd.items.selection = scopeItems?.[0]?.name || null;
          cmd.filters = [];
          cmd.actions = [getDefaultActionForScope(action.scope)];
        }
        break;
      case 'TOGGLE_HELP':
        state.helpPanelOpen = action.open;
        break;
      case 'UNDO':
        if (undoStack.length > 0) {
          redoStack.push(JSON.stringify(state));
          state = createReactiveState(JSON.parse(undoStack.pop()), debouncedSave);
          debouncedSave();
        }
        break;
      case 'REDO':
        if (redoStack.length > 0) {
          undoStack.push(JSON.stringify(state));
          state = createReactiveState(JSON.parse(redoStack.pop()), debouncedSave);
          debouncedSave();
        }
        break;
      case 'IMPORT_SETTINGS':
        pushUndo();
        const imported = {
          ...action.settings,
          commands: Array.isArray(action.settings?.commands)
            ? action.settings.commands.map((cmd) => normalizeCommandModel(cmd))
            : []
        };
        state = createReactiveState(imported, debouncedSave);
        if (state.commands.length > 0 && !state.editingCommandId) {
          state.editingCommandId = state.commands[0].id;
        }
        debouncedSave();
        break;
      case 'UPDATE_EDITING_COMMAND':
        if (cmd) Object.assign(cmd, action.updates);
        break;
      case 'UPDATE_EDITING_COMMAND_ITEMS':
        if (cmd) Object.assign(cmd.items, action.updates);
        break;
      case 'UPDATE_EDITING_COMMAND_REFINEMENT':
        if (cmd) Object.assign(cmd.refinement, action.updates);
        break;
      case 'ADD_FILTER':
        if (cmd) {
          const scopeFilters = editorSchema.filters[cmd.items.scope] || [];
          const firstFilter = scopeFilters[0];
          const type = firstFilter?.type || 'string';
          const ops = editorSchema.operators[type] || ['contains'];
          let defaultValue = '';
          if (type === 'select' && firstFilter?.options?.length > 0) {
            const firstOpt = firstFilter.options[0];
            defaultValue = firstOpt?.value ?? firstOpt ?? '';
          } else if (type === 'boolean') {
            defaultValue = 'true';
          }
          cmd.filters.push({
            name: firstFilter?.name || 'URL',
            type,
            operator: ops[0] || 'contains',
            value: defaultValue,
            join: 'and'
          });
        }
        break;
      case 'DELETE_FILTER':
        if (cmd) cmd.filters.splice(action.index, 1);
        break;
      case 'MOVE_FILTER':
        if (cmd) {
          const { index, direction } = action;
          const newIndex = index + direction;
          if (newIndex >= 0 && newIndex < cmd.filters.length) {
            const [removed] = cmd.filters.splice(index, 1);
            cmd.filters.splice(newIndex, 0, removed);
          }
        }
        break;
      case 'UPDATE_FILTER':
        if (cmd && cmd.filters[action.index]) {
          Object.assign(cmd.filters[action.index], action.updates);
        }
        break;
      case 'ADD_ACTION':
        if (cmd) {
          cmd.actions.push(getDefaultActionForScope(cmd.items.scope));
        }
        break;
      case 'DELETE_ACTION':
        if (cmd) cmd.actions.splice(action.index, 1);
        break;
      case 'MOVE_ACTION':
        if (cmd) {
          const { index, direction } = action;
          const newIndex = index + direction;
          if (newIndex >= 0 && newIndex < cmd.actions.length) {
            const [removed] = cmd.actions.splice(index, 1);
            cmd.actions.splice(newIndex, 0, removed);
          }
        }
        break;
      case 'UPDATE_ACTION':
        if (cmd && cmd.actions[action.index]) {
          Object.assign(cmd.actions[action.index], action.updates);
        }
        break;
      case 'UPDATE_COMMAND_TRIGGERS': {
        const targetCmd = state.commands.find(c => c.id === action.id);
        if (targetCmd) {
          if (!targetCmd.triggers) targetCmd.triggers = {};
          Object.assign(targetCmd.triggers, action.updates);
        }
        break;
      }
      case 'UPDATE_COMMAND_NAME': {
        const targetCmd = state.commands.find(c => c.id === action.id);
        if (targetCmd) targetCmd.name = action.name;
        break;
      }
    }

    const shouldNotify = (() => {
      if (action.type === 'UPDATE_FILTER') {
        return !(action.updates && Object.keys(action.updates).length === 1 && 'value' in action.updates);
      }
      if (action.type === 'UPDATE_ACTION') {
        return !(action.updates && Object.keys(action.updates).length === 1 && 'inputs' in action.updates);
      }
      if (action.type === 'UPDATE_COMMAND_NAME') return false;
      if (action.type === 'UPDATE_COMMAND_TRIGGERS') {
        const keys = action.updates ? Object.keys(action.updates) : [];
        const isTextOnly = keys.length > 0 && keys.every(k => k === 'omnibox' || k === 'shortcutSlot');
        return !isTextOnly;
      }
      return true;
    })();

    if (shouldNotify) notify();
  }
};
