import { store } from './store.js';
import { $, $$, createElement, fromHTML } from './dom-utils.js';
import { svgs } from './svgs.js';
import { renderCommandTable } from './command-table.js';
import { renderEditor } from './editor.js';
import { editorSchema } from './editor-schema.js';
import { helpPanelContent } from './help-content.js';
import { logger } from '../engine/logger.js';

// Helper functions for UI actions
function isCommandContextSingle(command) {
  const type = command.refinement?.type;
  if (type === 'First' || type === 'Last') return true;
  if (type === 'Random' && (command.refinement?.n || 1) === 1) return true;
  return false;
}

function isRandomRefinement(refinement) {
  return refinement?.type === 'Random';
}

function getScopePreviewLabel(scope, count) {
  const labels = {
    tab: count === 1 ? 'tab' : 'tabs',
    window: count === 1 ? 'window' : 'windows',
    group: count === 1 ? 'group' : 'groups',
    bookmark: count === 1 ? 'bookmark' : 'bookmarks',
    download: count === 1 ? 'download' : 'downloads',
    history: count === 1 ? 'history item' : 'history items',
    reading_list: count === 1 ? 'reading list item' : 'reading list items'
  };
  return labels[scope] || (count === 1 ? 'item' : 'items');
}

function formatActionLabel(action, isSingleContext = false) {
  if (!action) return '';

  const label = action.label || '';
  if (isSingleContext) return label;

  const multiLabelMap = {
    copy_url_to_clipboard: 'Copy URLs to Clipboard',
    copy_urls_to_clipboard: 'Copy URLs to Clipboard'
  };

  return multiLabelMap[action.id] || label;
}

function getAllowedActionsForSelection(scope, selection, isSingleContext = null) {
  const scopeActions = editorSchema.actions[scope] || [];
  const scopeItems = editorSchema.items[scope] || [];
  const selectedItem = scopeItems.find(i => i.name === selection);
  const isSingle = isSingleContext ?? (selectedItem?.multi === false);

  // Filter actions based on whether they're available for single/multi context
  return scopeActions.filter(action => {
    if (action.only === 'single' && !isSingle) return false;
    if (action.only === 'multi' && isSingle) return false;
    if (action.excludeSelections?.includes(selection)) return false;
    return true;
  });
}

function getDefaultInputsForAction(actionId) {
  // Find the action definition and return default inputs
  for (const scope of Object.keys(editorSchema.actions)) {
    const action = editorSchema.actions[scope].find(a => a.id === actionId);
    if (action?.inputs) {
      const defaults = {};
      for (const input of action.inputs) {
        if (input.default !== undefined) {
          defaults[input.key] = input.default;
        } else if (input.type === 'select' && input.options?.length > 0) {
          const first = input.options[0];
          defaults[input.key] = typeof first === 'object' && first !== null ? first.value : first;
        } else if (input.type === 'checkboxes') {
          defaults[input.key] = [];
        } else {
          defaults[input.key] = '';
        }
      }
      return defaults;
    }
  }
  return {};
}

// The main UI shell as a string
const appShell = `
<div id="app">
  <header>
    <div class="header-inner">
      <div class="header-left">
        <span class="header-logo">${svgs.headerLogo}</span>
      </div>
      <div class="header-right">
        <button class="btn-accent" id="help-menu-btn">Getting Started</button>
        <button class="btn-accent" id="theme-toggle-btn">Dark Mode</button>
        <div class="header-button-group">
          <button class="btn-accent" id="export-btn">Export</button>
          <button class="btn-accent" id="import-btn">Import</button>
        </div>
        <div class="header-button-group">
          <button class="btn-accent" id="undo-btn" title="Undo (Ctrl+Z)" disabled>
            ${svgs.undo} Undo
          </button>
          <button class="btn-accent" id="redo-btn" title="Redo (Ctrl+Shift+Z)" disabled>
            ${svgs.redo} Redo
          </button>
        </div>
      </div>
    </div>
  </header>

  <main>
    <div class="container">
      <div class="help-panel" id="help-panel">
        <button class="icon-button help-panel-close" id="help-panel-close" title="Close">${svgs.helpClose}</button>
        ${helpPanelContent}
      </div>

      <section class="command-section">
        <div class="command-table-frame">
          <div class="command-table" id="command-table">
            <div class="command-triggers-label">Triggers</div>
            <div class="command-row command-header-row">
              <div class="command-cell command-cell-editing"></div>
              <div class="command-cell command-cell-name">Name</div>
              <div class="command-cell command-cell-trigger command-trigger-start">Icon</div>
              <div class="command-cell command-cell-trigger">Shortcut</div>
              <div class="command-cell command-cell-trigger">Context</div>
              <div class="command-cell command-cell-trigger command-trigger-end">Omnibox</div>
              <div class="command-cell command-cell-actions"></div>
            </div>
          </div>
        </div>
        <div class="command-controls indent-row">
          <button class="btn-accent" id="add-command-btn">Add Command</button>
        </div>
      </section>

      <div class="command-editor" id="command-editor">
        <div class="logic-block">
          <div class="logic-label">Items</div>
          <div class="item-row item-row-wrap">
            <select id="items-scope-select"></select>
            <span id="items-options">
              <span class="chevron-sep">${svgs.chevron}</span>
              <select id="items-selection-select"></select>
              <label class="action-input-inline-label include-active-label" id="include-active-label">
                <input type="checkbox" id="include-active-checkbox">
                <span class="header-label">Include Active Tab</span>
              </label>
              <span id="item-options-container"></span>
            </span>
          </div>
        </div>

        <div class="logic-block" id="filters-block">
          <div class="logic-label">Filters</div>
          <div id="filters-container"></div>
          <button class="btn-accent" id="add-filter-btn">Add Filter</button>
          <div class="item-row limit-row">
            <span id="refinement-wrap">
              <span class="header-label">Select</span>
              <select id="refinement-select"></select>
              <label id="refinement-n-label" class="action-input-inline-label">
                <span class="header-label">Amount</span>
                <input type="number" id="refinement-n" class="form-input" min="1" value="1">
              </label>
            </span>
            <button type="button" class="btn-link preview-toggle-btn-wrap" id="preview-toggle-btn">Show Preview</button>
          </div>
          <div class="preview-content-inner" id="preview-content">
            <div class="preview-area" id="preview-area">
              <div class="preview-count" id="preview-count">0 items would be selected</div>
            </div>
            <button type="button" class="btn-link" id="preview-refresh-btn">Refresh</button>
          </div>
        </div>

        <div class="logic-block" id="actions-block">
          <div class="logic-label">Actions</div>
          <div>
            <div class="action-context-label" id="action-context-label">
              Showing actions available for <b>items</b>
            </div>
            <div id="actions-container"></div>
            <button class="btn-accent" id="add-action-btn">Add Action</button>
          </div>
          <div class="item-row action-confirm-row">
            <span class="header-label">Notification</span>
            <select id="execution-mode-select">
              <option value="silent">Just Run</option>
              <option value="notify">Run and Notify</option>
              <option value="confirm">Confirm First</option>
            </select>
            <label class="action-input-inline-label">
              <input type="checkbox" id="notify-on-error-checkbox" checked>
              <span class="header-label">Notify on Error</span>
            </label>
          </div>
        </div>

        <div class="actions-controls indent-row">
          <button id="run-command-btn">${svgs.run} Run Command Now</button>
        </div>
      </div>
    </div>
  </main>

  <input type="file" id="import-file-input" accept=".json">
</div>
`;

// State Actions Wrapper for UI
const uiActions = {
  getEditingCommand: () => store.getState().commands.find(c => c.id === store.getState().editingCommandId),
  selectCommand: (id) => store.dispatch({ type: 'SELECT_COMMAND', id }),
  moveCommand: (id, direction) => store.dispatch({ type: 'MOVE_COMMAND', id, direction }),
  deleteCommand: (id) => store.dispatch({ type: 'DELETE_COMMAND', id }),
  runCommand: async (command) => {
    const result = await chrome.runtime.sendMessage({ type: 'RUN_COMMAND', command });
    if (!result.success) logger.error('Command failed:', result.error);
  },
  updateHelpPanelState,
  renderEditor: () => renderEditor(store.getState(), uiActions, editorSchema),
  isCommandContextSingle,
  formatActionLabel,
  getAllowedActionsForSelection,
  getDefaultInputsForAction,
  refreshPreviewIfVisible,
  deleteFilter: (index) => store.dispatch({ type: 'DELETE_FILTER', index }),
  moveFilter: (index, direction) => store.dispatch({ type: 'MOVE_FILTER', index, direction }),
  deleteAction: (index) => store.dispatch({ type: 'DELETE_ACTION', index }),
  moveAction: (index, direction) => store.dispatch({ type: 'MOVE_ACTION', index, direction })
};

// Theme
function applyTheme(theme) {
  const isDark = theme === 'dark' ||
    (theme === 'system' && window.matchMedia('(prefers-color-scheme: dark)').matches);
  document.documentElement.setAttribute('data-theme', isDark ? 'dark' : 'light');
  const btn = $('theme-toggle-btn');
  if (btn) btn.textContent = isDark ? 'Light Mode' : 'Dark Mode';
}

async function initTheme() {
  let theme = 'system';
  try {
    const result = await chrome.storage.local.get('tifa_theme');
    if (result.tifa_theme) theme = result.tifa_theme;
  } catch { /* fallback to system */ }
  applyTheme(theme);
  return theme;
}

// Shell & UI building
function buildShell() {
  const existingApp = document.getElementById('app');
  const shell = fromHTML(appShell);
  
  if (existingApp) {
    existingApp.replaceWith(shell);
  } else {
    document.body.appendChild(shell);
  }

  // No special links.
}

// Preview
let previewVisible = false;
function refreshPreviewIfVisible() { if (previewVisible) refreshPreview(); }

function updateUndoRedoButtons() {
  const undoBtn = $('undo-btn');
  const redoBtn = $('redo-btn');
  if (undoBtn) undoBtn.disabled = store.getUndoCount() === 0;
  if (redoBtn) redoBtn.disabled = store.getRedoCount() === 0;
}

async function refreshPreview() {
  const command = uiActions.getEditingCommand();
  if (!command || command.items.scope === 'global') return;
  const previewArea = $('preview-area');
  const previewCount = $('preview-count');
  if (!previewArea || !previewCount) return;
  for (const el of $$('.preview-item, .preview-more', previewArea)) {
    el.remove();
  }
  previewCount.textContent = 'Loading...';
  try {
    const result = await chrome.runtime.sendMessage({ type: 'PREVIEW_ITEMS', itemsConfig: command.items, filters: command.filters, refinement: command.refinement });
    if (result.success) {
      const label = getScopePreviewLabel(command.items.scope, result.count, command.refinement);
      const randomSuffix = isRandomRefinement(command.refinement) ? ' (random)' : '';
      previewCount.replaceChildren(
        createElement('b', { children: [`${result.count} ${label}`] }),
        ` would be selected if run now${randomSuffix}`
      );
      for (const item of result.items) {
        previewArea.appendChild(createElement('div', { className: 'preview-item', children: [item.title || item.url] }));
      }
      if (result.hasMore) {
        const remaining = result.count - result.items.length;
        if (remaining > 0) {
          const moreLabel = getScopePreviewLabel(command.items.scope, remaining, command.refinement);
          previewArea.appendChild(createElement('div', { className: 'preview-more', children: [`and ${remaining} more ${moreLabel}...`] }));
        }
      }
    } else previewCount.textContent = `Error: ${result.error}`;
  } catch (error) { previewCount.textContent = `Error: ${error.message}`; }
}

function updateHelpPanelState() {
  const panel = $('help-panel');
  if (!panel) return;
  const isOpen = store.getState().helpPanelOpen !== false;
  if (isOpen) panel.classList.remove('hidden');
  else panel.classList.add('hidden');
  const btn = $('help-menu-btn');
  if (btn) btn.textContent = isOpen ? 'Hide Getting Started' : 'Getting Started';
}

// Event listeners
function setupEventListeners() {
  $('add-command-btn').addEventListener('click', () => store.dispatch({ type: 'ADD_COMMAND' }));
  $('items-scope-select').addEventListener('change', (e) => store.dispatch({ type: 'UPDATE_COMMAND_SCOPE', scope: e.target.value }));
  
  $('items-selection-select').addEventListener('change', (e) => {
    store.dispatch({ type: 'UPDATE_EDITING_COMMAND_ITEMS', updates: { selection: e.target.value } });
  });
  
  $('include-active-checkbox').addEventListener('change', (e) => {
    store.dispatch({ type: 'UPDATE_EDITING_COMMAND_ITEMS', updates: { includeActiveTab: e.target.checked } });
  });
  
  $('add-filter-btn').addEventListener('click', () => {
    store.dispatch({ type: 'ADD_FILTER' });
  });
  
  $('add-action-btn').addEventListener('click', () => {
    store.dispatch({ type: 'ADD_ACTION' });
  });
  
  $('refinement-select').addEventListener('change', (e) => {
    store.dispatch({ type: 'UPDATE_EDITING_COMMAND_REFINEMENT', updates: { type: e.target.value } });
  });
  
  $('refinement-n').addEventListener('change', (e) => {
    const n = Math.max(1, parseInt(e.target.value, 10) || 1);
    e.target.value = n;
    store.dispatch({ type: 'UPDATE_EDITING_COMMAND_REFINEMENT', updates: { n } });
  });
  
  $('preview-toggle-btn').addEventListener('click', () => {
    previewVisible = !previewVisible;
    $('preview-content').style.display = previewVisible ? 'block' : 'none';
    $('preview-toggle-btn').textContent = previewVisible ? 'Hide Preview' : 'Show Preview';
    if (previewVisible) refreshPreview();
  });
  
  $('preview-refresh-btn').addEventListener('click', refreshPreview);
  
  $('execution-mode-select').addEventListener('change', (e) => {
    store.dispatch({ type: 'UPDATE_EDITING_COMMAND', updates: { executionMode: e.target.value } });
  });

  $('notify-on-error-checkbox').addEventListener('change', (e) => {
    store.dispatch({ type: 'UPDATE_EDITING_COMMAND', updates: { notifyOnError: e.target.checked } });
  });
  
  $('run-command-btn').addEventListener('click', () => {
    const command = uiActions.getEditingCommand();
    if (command) uiActions.runCommand(command);
  });
  
  $('undo-btn').addEventListener('click', () => store.dispatch({ type: 'UNDO' }));
  $('redo-btn').addEventListener('click', () => store.dispatch({ type: 'REDO' }));
  
  $('help-panel-close').addEventListener('click', () => store.dispatch({ type: 'TOGGLE_HELP', open: false }));
  $('help-menu-btn').addEventListener('click', () => {
    const isOpen = store.getState().helpPanelOpen !== false;
    store.dispatch({ type: 'TOGGLE_HELP', open: !isOpen });
  });
  
  $('export-btn').addEventListener('click', () => {
    const data = JSON.stringify(store.getState(), null, 2);
    const blob = new Blob([data], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = createElement('a', { href: url, download: 'tifa-settings.json' });
    a.click();
    URL.revokeObjectURL(url);
  });
  
  $('import-btn').addEventListener('click', () => {
    $('import-file-input').click();
  });
  
  $('import-file-input').addEventListener('change', async (e) => {
    const file = e.target.files[0];
    if (!file) return;
    try {
      const text = await file.text();
      const imported = JSON.parse(text);
      if (imported.commands) store.dispatch({ type: 'IMPORT_SETTINGS', settings: imported });
    } catch (err) { alert('Import failed: ' + err.message); }
    e.target.value = '';
  });
  
  let currentTheme = 'system';
  initTheme().then(t => { currentTheme = t; });

  $('theme-toggle-btn').addEventListener('click', () => {
    const isDark = document.documentElement.getAttribute('data-theme') === 'dark';
    currentTheme = isDark ? 'light' : 'dark';
    applyTheme(currentTheme);
    try { chrome.storage.local.set({ tifa_theme: currentTheme }); } catch { /* ignore */ }
  });

  window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', () => {
    if (currentTheme === 'system') applyTheme('system');
  });

  document.addEventListener('keydown', (e) => {
    if (e.ctrlKey || e.metaKey) {
      if (e.key === 'z' && !e.shiftKey) { e.preventDefault(); store.dispatch({ type: 'UNDO' }); }
      else if ((e.key === 'z' && e.shiftKey) || e.key === 'y') { e.preventDefault(); store.dispatch({ type: 'REDO' }); }
    }
  });
}

// Initialize
async function init() {
  buildShell();
  const state = await store.init();
  
  store.subscribe((newState) => {
    renderCommandTable(newState, uiActions);
    updateHelpPanelState();
    updateUndoRedoButtons();
  });

  updateHelpPanelState();
  updateUndoRedoButtons();
  renderCommandTable(state, uiActions);
  setupEventListeners();
}

init();
