import { $, createElement, fromHTML } from './dom-utils.js';
import { svgs } from './svgs.js';
import { store } from './store.js';

let pendingActionSelectFocusIndex = null;

function updateMoveButtons(container) {
  if (!container) return;
  const rows = [...container.children].filter((row) =>
    row.querySelector('[title="Move up"]') || row.querySelector('[title="Move down"]')
  );
  const hasMultiple = rows.length > 1;
  for (const [i, row] of rows.entries()) {
    const upBtn = row.querySelector('[title="Move up"]');
    const downBtn = row.querySelector('[title="Move down"]');
    if (upBtn) {
      upBtn.disabled = !hasMultiple || i === 0;
      upBtn.style.display = hasMultiple ? '' : 'none';
    }
    if (downBtn) {
      downBtn.disabled = !hasMultiple || i === rows.length - 1;
      downBtn.style.display = hasMultiple ? '' : 'none';
    }
  }
}


export function renderEditor(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const scopeSelect = $('items-scope-select');
  if (scopeSelect) {
    scopeSelect.innerHTML = '';
    for (const key of Object.keys(schema.scopeLabels)) {
      scopeSelect.appendChild(createElement('option', {
        value: key,
        selected: command.items.scope === key,
        children: [schema.scopeLabels[key]]
      }));
    }
  }

  updateSelectionOptions(state, actions, schema);
  updateIncludeActiveCheckbox(state, actions, schema);
  updateItemOptions(state, actions, schema);

  const isGlobal = command.items.scope === 'global';
  const scopeItems = schema.items[command.items.scope] || [];
  const hasMultipleOptions = scopeItems.length > 1;

  const filtersBlock = $('filters-block');
  if (filtersBlock) filtersBlock.style.display = isGlobal ? 'none' : '';
  const itemsOptions = $('items-options');
  if (itemsOptions) itemsOptions.style.display = (isGlobal || !hasMultipleOptions) ? 'none' : '';

  renderFilters(state, actions, schema);
  renderRefinement(state, actions, schema);
  renderActions(state, actions, schema);

  const executionModeSelect = $('execution-mode-select');
  if (executionModeSelect) {
    executionModeSelect.value = command.executionMode || 'silent';
  }
  const notifyOnErrorCheckbox = $('notify-on-error-checkbox');
  if (notifyOnErrorCheckbox) {
    notifyOnErrorCheckbox.checked = command.notifyOnError !== false;
  }
  updateActionContextLabel(state, actions, schema);
}

export function updateSelectionOptions(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const selectionSelect = $('items-selection-select');
  if (!selectionSelect) return;
  selectionSelect.innerHTML = '';
  const scopeItems = schema.items[command.items.scope];
  if (scopeItems) {
    for (const item of scopeItems) {
      selectionSelect.appendChild(createElement('option', {
        value: item.name,
        selected: command.items.selection === item.name,
        children: [item.name]
      }));
    }
  }
}

export function updateIncludeActiveCheckbox(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const scopeItems = schema.items[command.items.scope];
  const selectedItem = scopeItems?.find(i => i.name === command.items.selection);

  const label = $('include-active-label');
  const checkbox = $('include-active-checkbox');
  if (selectedItem && selectedItem.includeActiveTab) {
    if (label) label.style.display = 'inline-flex';
    if (checkbox) checkbox.checked = command.items.includeActiveTab !== false;
  } else {
    if (label) label.style.display = 'none';
  }
}

export function updateItemOptions(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const container = $('item-options-container');
  if (!container) return;
  container.innerHTML = '';

  const scopeItems = schema.items[command.items.scope] || [];
  const selectedItem = scopeItems.find(i => i.name === command.items.selection);

  if (!selectedItem?.options) return;

  const currentOpts = command.items.options || {};
  const elements = [];

  for (const opt of selectedItem.options) {
    const value = currentOpts[opt.key] ?? opt.default;

    if (opt.type === 'toggle') {
      const checked = value !== false;
      const label = createElement('label', {
        className: 'inline-label',
        children: [
          createElement('input', { type: 'checkbox', checked }),
          ` ${opt.label}`
        ]
      });
      const checkbox = label.querySelector('input');
      checkbox.addEventListener('change', (e) => {
        const options = { ...(command.items.options || {}) };
        options[opt.key] = e.target.checked;
        store.dispatch({ type: 'UPDATE_EDITING_COMMAND_ITEMS', updates: { options } });
      });
      elements.push({ el: label, key: opt.key });
    } else {
      const shouldHide = opt.showWhen && currentOpts[opt.showWhen.key] === false;
      const label = createElement('label', {
        className: 'action-input-inline-label',
        children: [
          createElement('span', { className: 'header-label', children: [opt.label] }),
          createElement('input', {
            type: opt.type === 'number' ? 'number' : 'text',
            className: 'form-input',
            value: String(value ?? ''),
            ...(opt.min !== undefined ? { min: String(opt.min) } : {}),
            ...(opt.step !== undefined ? { step: String(opt.step) } : {})
          })
        ]
      });
      if (shouldHide) label.style.display = 'none';
      label.dataset.showWhenKey = opt.showWhen?.key || '';
      const input = label.querySelector('input');
      input.addEventListener('change', (e) => {
        const options = { ...(command.items.options || {}) };
        const raw = e.target.value;
        options[opt.key] = opt.type === 'number' ? (raw === '' ? undefined : parseInt(raw, 10) || 0) : raw;
        store.dispatch({ type: 'UPDATE_EDITING_COMMAND_ITEMS', updates: { options } });
      });
      elements.push({ el: label, key: opt.key });
    }
  }

  for (const { el } of elements) {
    container.appendChild(el);
  }
}

export function renderFilters(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const container = $('filters-container');
  if (!container) return;
  container.innerHTML = '';
  for (const [i, filter] of command.filters.entries()) {
    container.appendChild(createFilterRow(filter, i, state, actions, schema));
  }

  updateFilterMoveButtons();
}

export function createFilterRow(filter, index, state, actions, schema) {
  const command = actions.getEditingCommand();
  const isLast = index === command.filters.length - 1;
  const scopeFilters = schema.filters[command.items.scope] || [];
  const filterDef = scopeFilters.find(f => f.name === filter.name);
  
  const getOperatorsForType = (type) => {
    const ops = schema.operators;
    if (ops && typeof ops === 'object' && !Array.isArray(ops)) return ops[type] || ['contains'];
    const operatorDef = Array.isArray(ops) ? ops.find(o => o.type === type) : null;
    return operatorDef?.operators || ['contains'];
  };

  const getDefaultOperator = (type) => getOperatorsForType(type)[0] || 'contains';
  const operators = getOperatorsForType(filterDef?.type || 'string');

  let valueInputHtml = '';
  if (filterDef?.type === 'boolean') {
    valueInputHtml = `
      <select class="filter-value-select">
        <option value="true" ${filter.value === 'true' ? 'selected' : ''}>Yes</option>
        <option value="false" ${filter.value !== 'true' ? 'selected' : ''}>No</option>
      </select>
    `;
  } else if (filterDef?.type === 'select' && filterDef.options) {
    valueInputHtml = `
      <select class="filter-value-select">
        ${filterDef.options.map(opt => {
          const val = opt?.value ?? opt;
          const lab = opt?.label ?? opt;
          return `<option value="${val}" ${filter.value === val ? 'selected' : ''}>${lab}</option>`;
        }).join('')}
      </select>
    `;
  } else if (filterDef?.type === 'datetime') {
    const emptyClass = !filter.value ? ' input-invalid' : '';
    valueInputHtml = `<div class="col-fill"><input type="datetime-local" class="form-input filter-value-input${emptyClass}" value="${filter.value || ''}"></div>`;
  } else {
    const emptyClass = !filter.value ? ' input-invalid' : '';
    valueInputHtml = `<div class="col-fill"><input type="${filterDef?.type === 'number' ? 'number' : 'text'}" class="form-input filter-value-input${emptyClass}" value="${filter.value || ''}"></div>`;
  }

    const html = `
      <div class="item-row item-row-wrap">
        <select class="filter-name-select">
          ${scopeFilters.map(f => `<option value="${f.name}" ${filter.name === f.name ? 'selected' : ''}>${f.name}</option>`).join('')}
        </select>
        <span class="chevron-sep">${svgs.chevron}</span>

        ${operators.length > 1 ? `
          <select class="filter-operator-select">
            ${operators.map(op => `<option value="${op}" ${filter.operator === op ? 'selected' : ''}>${op}</option>`).join('')}
          </select>
        ` : ''}

      ${valueInputHtml}

      <div class="filter-row-controls">
        <div class="icon-button-group">
          <button type="button" class="icon-button move-up-btn" title="Move up">${svgs.moveUp}</button>
          <button type="button" class="icon-button move-down-btn" title="Move down">${svgs.moveDown}</button>
          <button type="button" class="icon-button icon-button-danger delete-btn" title="Delete">${svgs.delete}</button>
        </div>

        ${!isLast ? `
          <select class="filter-join-select">
            <option value="and" ${filter.join !== 'or' ? 'selected' : ''}>And...</option>
            <option value="or" ${filter.join === 'or' ? 'selected' : ''}>Or...</option>
          </select>
        ` : (command.filters.length > 1 ? '<span class="filter-join-spacer" aria-hidden="true"></span>' : '')}
      </div>
    </div>
  `;

  const row = fromHTML(html);
  const q = (s) => row.querySelector(s);

  q('.filter-name-select').onchange = (e) => {
    const name = e.target.value;
    const def = scopeFilters.find(f => f.name === name);
    const type = def?.type || 'string';
    let defaultValue = '';
    if (type === 'select' && def?.options?.length > 0) {
      const firstOpt = def.options[0];
      defaultValue = firstOpt?.value ?? firstOpt ?? '';
    } else if (type === 'boolean') {
      defaultValue = 'true';
    }
    store.dispatch({
      type: 'UPDATE_FILTER',
      index,
      updates: { name, type, operator: getDefaultOperator(type), value: defaultValue }
    });
  };

  if (operators.length > 1) {
    q('.filter-operator-select').onchange = (e) => {
      store.dispatch({ type: 'UPDATE_FILTER', index, updates: { operator: e.target.value } });
    };
  }

  const valueEl = q('.filter-value-select') || q('.filter-value-input');
  if (valueEl) {
    const eventType = valueEl.tagName === 'SELECT' || valueEl.type === 'datetime-local' ? 'change' : 'input';
    valueEl.addEventListener(eventType, (e) => {
      store.dispatch({ type: 'UPDATE_FILTER', index, updates: { value: e.target.value } });
      if (valueEl.tagName !== 'SELECT') {
        valueEl.classList.toggle('input-invalid', !e.target.value);
      }
    });
  }

  q('.move-up-btn').onclick = () => store.dispatch({ type: 'MOVE_FILTER', index, direction: -1 });
  q('.move-down-btn').onclick = () => store.dispatch({ type: 'MOVE_FILTER', index, direction: 1 });
  q('.delete-btn').onclick = () => store.dispatch({ type: 'DELETE_FILTER', index });

  if (!isLast) {
    q('.filter-join-select').onchange = (e) => {
      store.dispatch({ type: 'UPDATE_FILTER', index, updates: { join: e.target.value } });
    };
  }

  return row;
}

export function updateFilterMoveButtons() {
  updateMoveButtons($('filters-container'));
}

export function renderRefinement(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const scopeItems = schema.items[command.items.scope] || [];
  const selectedItem = scopeItems.find(i => i.name === command.items.selection);
  const isMulti = selectedItem ? selectedItem.multi !== false : true;

  const wrap = $('refinement-wrap');
  const previewBtn = $('preview-toggle-btn');
  if (wrap) wrap.style.display = isMulti ? '' : 'none';
  if (previewBtn) previewBtn.style.marginLeft = isMulti ? '' : '0';

  const type = command.refinement?.type || 'All';

  const select = $('refinement-select');
  if (select) {
    select.innerHTML = '';
    for (const r of schema.filters.Refinement) {
      select.appendChild(createElement('option', {
        value: r.name,
        selected: type === r.name,
        children: [r.name]
      }));
    }
  }

  const shouldShowCount = type !== 'All';
  const nLabel = $('refinement-n-label');
  const nInput = $('refinement-n');
  if (nLabel) {
    nLabel.style.display = shouldShowCount ? 'inline-flex' : 'none';
  }
  if (nInput) {
    nInput.value = command.refinement?.n || 1;
  }
}

export function renderActions(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const container = $('actions-container');
  if (!container) return;
  container.innerHTML = '';
  for (const [i, action] of command.actions.entries()) {
    container.appendChild(createActionRow(action, i, state, actions, schema));
  }

  if (pendingActionSelectFocusIndex !== null) {
    const selects = container.querySelectorAll('.action-select');
    const target = selects[pendingActionSelectFocusIndex];
    if (target) target.focus();
    pendingActionSelectFocusIndex = null;
  }

  updateActionMoveButtons();
}

export function createActionRow(action, index, state, actions, schema) {
  const command = actions.getEditingCommand();
  const isSingleContext = (() => {
      const items = schema.items[command.items.scope];
      const sel = items?.find(i => i.name === command.items.selection);
      return actions.isCommandContextSingle(command) || sel?.multi === false;
  })();
  const scopeActions = actions.getAllowedActionsForSelection(command.items.scope, command.items.selection, isSingleContext);

  const actionId = action.id || action.name;
  const actionDef = scopeActions.find(a => a.id === actionId);
  const unknownActionOption = (!actionDef && actionId)
    ? `<option value="${actionId}" selected>Unknown action (${actionId})</option>`
    : '';
  const isStackedInputLayout = actionDef?.layout === 'stacked';
  const actionRowMainClass = isStackedInputLayout
    ? 'item-row action-row-main action-row-main-stacked'
    : 'item-row action-row-main';

  // Build inputs HTML
  let inputsHtml = '';
  if (actionDef?.inputs) {
    for (const inputDef of actionDef.inputs) {
      if (inputDef.type === 'select' && inputDef.options) {
        inputsHtml += `
          <label class="action-input-inline-label action-input-inline-label-select">
            <span class="header-label">${inputDef.label}</span>
            <select class="action-input" data-key="${inputDef.key}">
              ${inputDef.options.map(opt => {
                const value = typeof opt === 'object' && opt !== null ? opt.value : opt;
                const label = typeof opt === 'object' && opt !== null ? (opt.label ?? opt.value) : opt;
                return `<option value="${value}" ${action.inputs?.[inputDef.key] === value ? 'selected' : ''}>${label}</option>`;
              }).join('')}
            </select>
          </label>
        `;
      } else if (inputDef.type === 'boolean') {
        const checked = Boolean(action.inputs?.[inputDef.key] ?? inputDef.default ?? false);
        const hidden = inputDef.showWhen && action.inputs?.[inputDef.showWhen.key] !== inputDef.showWhen.value;
        inputsHtml += `
          <label class="action-checkbox"${hidden ? ' style="display:none"' : ''} ${inputDef.showWhen ? `data-show-when-key="${inputDef.showWhen.key}" data-show-when-value="${inputDef.showWhen.value}"` : ''}>
            <input type="checkbox"
                   class="action-input action-input-checkbox"
                   data-key="${inputDef.key}"
                   ${checked ? 'checked' : ''}>
            <span class="header-label">${inputDef.label}</span>
          </label>
        `;
      } else if (inputDef.type !== 'textarea' && inputDef.type !== 'checkboxes') {
        const value = action.inputs?.[inputDef.key] ?? inputDef.default ?? '';
        const isEmpty = inputDef.required && (value === '' || value === undefined || value === null);
        const hidden = inputDef.showWhen && action.inputs?.[inputDef.showWhen.key] !== inputDef.showWhen.value;
        inputsHtml += `
          <label class="action-input-inline-label"${hidden ? ' style="display:none"' : ''} ${inputDef.showWhen ? `data-show-when-key="${inputDef.showWhen.key}" data-show-when-value="${inputDef.showWhen.value}"` : ''}>
            <span class="header-label">${inputDef.label}</span>
            <input type="${inputDef.type === 'number' ? 'number' : 'text'}"
                   class="form-input action-input${isEmpty ? ' input-invalid' : ''}"
                   data-key="${inputDef.key}"
                   ${inputDef.required ? 'data-required="true"' : ''}
                   value="${value}"
                   ${inputDef.min !== undefined ? `min="${inputDef.min}"` : ''}
                   ${inputDef.max !== undefined ? `max="${inputDef.max}"` : ''}
                   ${inputDef.step !== undefined ? `step="${inputDef.step}"` : ''}>
          </label>
        `;
      }
    }
  }

  const checkboxInputs = (actionDef?.inputs || []).filter(i => i.type === 'checkboxes');
  let checkboxesHtml = '';
  if (checkboxInputs.length > 0) {
    checkboxesHtml = checkboxInputs.map(inputDef => {
      const selected = Array.isArray(action.inputs?.[inputDef.key]) ? action.inputs[inputDef.key] : [];
      return `
        <div class="item-row action-row-checkboxes">
          <div class="col-fill">
            <div class="action-checkbox-group">
              ${inputDef.label ? `<div class="action-input-label">${inputDef.label}</div>` : ''}
              <div class="action-checkbox-list">
                ${inputDef.options.map(opt => {
                  const val = opt.value ?? opt;
                    const lab = opt.label ?? opt;
                    return `
                      <label class="action-checkbox">
                        <input type="checkbox" class="action-checkbox-input" data-key="${inputDef.key}" data-value="${val}" ${selected.includes(val) ? 'checked' : ''}>
                        ${lab}
                      </label>
                    `;
                }).join('')}
              </div>
            </div>
          </div>
        </div>
      `;
    }).join('');
  }

  const textareaInputs = (actionDef?.inputs || []).filter(i => i.type === 'textarea');
  let textareasHtml = '';
  if (textareaInputs.length > 0) {
    textareasHtml = textareaInputs.map(inputDef => {
      const value = action.inputs?.[inputDef.key] || '';
      const isEmpty = inputDef.required && !value;
      return `
        <div class="action-row-textarea">
          <label class="action-textarea-label header-label">${inputDef.label}</label>
          <textarea class="form-input action-input action-textarea${isEmpty ? ' input-invalid' : ''}"
                    data-key="${inputDef.key}"
                    ${inputDef.required ? 'data-required="true"' : ''}>${value}</textarea>
        </div>
      `;
    }).join('');
  }

  const html = `
    <div class="action-row">
      <div class="${actionRowMainClass}">
        <select class="action-select">
          ${unknownActionOption}
          ${scopeActions.map(a => `<option value="${a.id}" ${actionId === a.id ? 'selected' : ''}>${actions.formatActionLabel(a, isSingleContext)}</option>`).join('')}
        </select>
        ${inputsHtml}
        <div class="icon-button-group">
          <button type="button" class="icon-button move-up-btn" title="Move up">${svgs.moveUp}</button>
          <button type="button" class="icon-button move-down-btn" title="Move down">${svgs.moveDown}</button>
          <button type="button" class="icon-button icon-button-danger delete-btn" title="Delete" ${command.actions.length <= 1 ? 'style="display:none"' : ''}>${svgs.delete}</button>
        </div>
      </div>
      ${checkboxesHtml}
      ${textareasHtml}
    </div>
  `;

  const row = fromHTML(html);
  const q = (s) => row.querySelector(s);
  const qq = (s) => [...row.querySelectorAll(s)];

  q('.action-select').onchange = (e) => {
    pendingActionSelectFocusIndex = index;
    const name = e.target.value;
    store.dispatch({ type: 'UPDATE_ACTION', index, updates: { id: name, inputs: actions.getDefaultInputsForAction(name) } });
  };

  for (const el of qq('.action-input')) {
    const eventType = el.tagName === 'SELECT' || el.type === 'checkbox' ? 'change' : 'input';
    el.addEventListener(eventType, (e) => {
      const inputs = { ...(action.inputs || {}) };
      inputs[el.dataset.key] = el.type === 'checkbox' ? e.target.checked : e.target.value;
      store.dispatch({ type: 'UPDATE_ACTION', index, updates: { inputs } });
      // Toggle invalid state for required fields
      if (el.dataset.required) {
        el.classList.toggle('input-invalid', !e.target.value);
      }
      // Auto-resize textareas
      if (el.tagName === 'TEXTAREA') {
        el.style.height = 'auto';
        el.style.height = el.scrollHeight + 'px';
      }
      // Toggle conditional visibility
      if (el.tagName === 'SELECT') {
        for (const label of qq('[data-show-when-key]')) {
          if (label.dataset.showWhenKey === el.dataset.key) {
            label.style.display = e.target.value === label.dataset.showWhenValue ? '' : 'none';
          }
        }
      }
    });
  }

  // Initialize textarea heights
  for (const el of qq('.action-textarea')) {
    el.style.height = 'auto';
    el.style.height = Math.max(el.scrollHeight, 60) + 'px';
  }

  for (const el of qq('.action-checkbox-input')) {
    el.onchange = (e) => {
      const key = el.dataset.key;
      const val = el.dataset.value;
      const current = Array.isArray(action.inputs?.[key]) ? action.inputs[key] : [];
      const next = e.target.checked ? [...new Set([...current, val])] : current.filter(v => v !== val);
      const inputs = { ...(action.inputs || {}) };
      inputs[key] = next;
      store.dispatch({ type: 'UPDATE_ACTION', index, updates: { inputs } });
    };
  }

  q('.move-up-btn').onclick = () => store.dispatch({ type: 'MOVE_ACTION', index, direction: -1 });
  q('.move-down-btn').onclick = () => store.dispatch({ type: 'MOVE_ACTION', index, direction: 1 });
  q('.delete-btn').onclick = () => store.dispatch({ type: 'DELETE_ACTION', index });

  return row;
}

export function updateActionMoveButtons() {
  updateMoveButtons($('actions-container'));
}

export function updateActionContextLabel(state, actions, schema) {
  const command = actions.getEditingCommand();
  if (!command) return;

  const label = $('action-context-label');
  if (!label) return;
  const scopeItems = schema.items[command.items.scope];
  const selectedItem = scopeItems?.find(i => i.name === command.items.selection);

  const isSingleByRefinement = actions.isCommandContextSingle(command);
  const effectiveSingle = isSingleByRefinement || selectedItem?.multi === false;

  let itemType;
  if (command.items.scope === 'global') {
    itemType = 'global actions';
  } else {
    const scope = command.items.scope.toLowerCase().replace(/_/g, ' ');
    const isSpecialNoun = scope === 'history' || scope === 'reading list';
    
    if (effectiveSingle) {
      const singular = scope.replace(/s$/, '');
      itemType = isSpecialNoun ? `a single ${scope} item` : `a single ${singular}`;
    } else {
      const plural = scope.endsWith('s') ? scope : `${scope}s`;
      itemType = isSpecialNoun ? `multiple ${scope} items` : `multiple ${plural}`;
    }
  }

  label.innerHTML = `Showing actions available for <b>${itemType}</b>`;
}
