import { items as ITEM_DEFS } from '../defs/item-defs.js';
import { filters as FILTER_DEFS } from '../defs/filter-defs.js';
import { filterOperators } from '../defs/filters-operations.js';
import { actions as ACTION_DEFS } from '../defs/action-defs.js';

const scopeKeyToLabel = {
  tab: 'Tabs',
  window: 'Windows',
  group: 'Tab Groups',
  bookmark: 'Bookmarks',
  download: 'Downloads',
  history: 'History',
  reading_list: 'Reading List',
  global: 'Global'
};

function toUiItems(key) {
  const defs = ITEM_DEFS[key] || [];

  const includeActiveEligible = new Set([
    'In Current Window',
    'In All Windows',
    'Left of Active Tab',
    'Right of Active Tab',
    'Highlighted'
  ]);

  const singletonsByScope = {
    tab: new Set(['Active Tab']),
    window: new Set(['Focused Window']),
    group: new Set(['Containing Active Tab'])
  };
  const singletonSet = singletonsByScope[key] || new Set();

  return defs
    .filter((d) => !(key === 'bookmark' && d.name === 'In Folder'))
    .map((d) => ({
      name: d.name,
      includeActiveTab: key === 'tab' && includeActiveEligible.has(d.name),
      multi: !singletonSet.has(d.name),
      ...(d.options ? { options: d.options } : {})
    }));
}

function toUiFilters(key) {
  const defs = FILTER_DEFS[key] || [];

  return defs.map((d) => {
    if (Array.isArray(d.type)) {
      return { name: d.name, type: 'select', options: d.type };
    }
    return { name: d.name, type: d.type || 'string' };
  });
}

function getUiActionInputs(action) {
  return Array.isArray(action.inputs) ? action.inputs : null;
}

function toUiActions(key) {
  const defs = ACTION_DEFS[key] || [];
  return defs.map((a) => {
    // Actions with minItems: 1 can work on single items, so they're not "multi-only"
    const isMultiOnly = a.selection === 'group' && (a.minItems ?? 2) >= 2;
    const only = a.selection === 'single' ? 'single' : isMultiOnly ? 'multi' : undefined;
    const inputs = getUiActionInputs(a);
    return {
      id: a.id,
      label: a.label,
      ...(only ? { only } : {}),
      ...(inputs ? { inputs } : {}),
      ...(a.layout ? { layout: a.layout } : {}),
      ...(a.excludeSelections ? { excludeSelections: a.excludeSelections } : {})
    };
  });
}

const operators = Object.fromEntries(
  Object.entries(filterOperators).map(([type, ops]) => [type, Object.keys(ops)])
);

const refinements = [
  { name: 'All' },
  { name: 'First' },
  { name: 'Last' },
  { name: 'Random' }
];

export const editorSchema = {
  items: Object.fromEntries(Object.keys(scopeKeyToLabel).filter(k => k !== 'global').map((key) => [key, toUiItems(key)])),
  filters: {
    ...Object.fromEntries(Object.keys(scopeKeyToLabel).filter(k => k !== 'global').map((key) => [key, toUiFilters(key)])),
    Refinement: refinements
  },
  actions: Object.fromEntries(Object.keys(scopeKeyToLabel).map((key) => [key, toUiActions(key)])),
  operators,
  scopeLabels: scopeKeyToLabel
};





