import { $, $$, fromHTML } from './dom-utils.js';
import { svgs } from './svgs.js';
import { store } from './store.js';

export function createCommandRow(command, state, actions) {
  const isEditing = command.id === state.editingCommandId;
  const triggers = command.triggers || {};

  const html = `
    <div class="command-row command-body-row ${isEditing ? 'is-editing' : ''}" data-id="${command.id}">
      <div class="command-cell command-cell-editing">
        ${isEditing ? 'Editing' : '<button class="btn-link edit-btn">Edit</button>'}
      </div>
      
      <div class="command-cell command-cell-name">
        <input type="text" class="command-name-input" value="${command.name || ''}">
      </div>

      <div class="command-cell command-cell-trigger command-trigger-start">
        <input type="checkbox" class="icon-trigger-checkbox" ${triggers.icon ? 'checked' : ''}>
      </div>

      <div class="command-cell command-cell-trigger">
        <select class="shortcut-slot-select">
          <option value="" ${!triggers.shortcutSlot ? 'selected' : ''}>-</option>
          ${Array.from({ length: 26 }, (_, i) => String.fromCharCode(65 + i))
            .map(letter => `<option value="${letter}" ${triggers.shortcutSlot === letter ? 'selected' : ''}>Slot ${letter}</option>`)
            .join('')}
        </select>
      </div>

      <div class="command-cell command-cell-trigger">
        <input type="checkbox" class="context-trigger-checkbox" ${triggers.context ? 'checked' : ''}>
      </div>

      <div class="command-cell command-cell-trigger command-trigger-end">
        <input type="text" class="omnibox-input" value="${triggers.omnibox || ''}">
      </div>

      <div class="command-cell command-cell-actions">
        <div class="icon-button-group">
          <button type="button" class="icon-button run-button" title="Run">${svgs.run}</button>
          <span class="move-delete-group">
            <button type="button" class="icon-button move-up-btn" title="Move up">${svgs.moveUp}</button>
            <button type="button" class="icon-button move-down-btn" title="Move down">${svgs.moveDown}</button>
            <button type="button" class="icon-button icon-button-danger delete-btn" title="Delete">${svgs.delete}</button>
          </span>
        </div>
      </div>
    </div>
  `;

  const row = fromHTML(html);

  // Wire up events
  const q = (s) => row.querySelector(s);

  if (!isEditing) {
    q('.edit-btn').onclick = () => store.dispatch({ type: 'SELECT_COMMAND', id: command.id });
  }

  q('.command-name-input').oninput = (e) => {
    store.dispatch({ type: 'UPDATE_COMMAND_NAME', id: command.id, name: e.target.value });
    updateContextCheckbox(row, command);
  };

  q('.icon-trigger-checkbox').onchange = (e) => {
    if (e.target.checked) {
      for (const c of state.commands) {
        if (c.id !== command.id && c.triggers?.icon) {
          store.dispatch({ type: 'UPDATE_COMMAND_TRIGGERS', id: c.id, updates: { icon: false } });
        }
      }
    }
    store.dispatch({ type: 'UPDATE_COMMAND_TRIGGERS', id: command.id, updates: { icon: e.target.checked } });
  };

  q('.shortcut-slot-select').onchange = (e) => {
    store.dispatch({ type: 'UPDATE_COMMAND_TRIGGERS', id: command.id, updates: { shortcutSlot: e.target.value || null } });
    updateShortcutDisabledStates(store.getState());
  };

  const contextCheck = q('.context-trigger-checkbox');
  contextCheck.onchange = (e) => {
    store.dispatch({ type: 'UPDATE_COMMAND_TRIGGERS', id: command.id, updates: { context: e.target.checked } });
  };
  updateContextCheckbox(row, command);

  q('.omnibox-input').oninput = (e) => {
    store.dispatch({ type: 'UPDATE_COMMAND_TRIGGERS', id: command.id, updates: { omnibox: e.target.value || null } });
    updateOmniboxHighlights(store.getState());
  };

  q('.run-button').onclick = () => actions.runCommand(command);
  q('.move-up-btn').onclick = () => store.dispatch({ type: 'MOVE_COMMAND', id: command.id, direction: -1 });
  q('.move-down-btn').onclick = () => store.dispatch({ type: 'MOVE_COMMAND', id: command.id, direction: 1 });
  q('.delete-btn').onclick = () => store.dispatch({ type: 'DELETE_COMMAND', id: command.id });

  return row;
}

export function updateContextCheckbox(row, command) {
  const checkbox = row.querySelector('.context-trigger-checkbox');
  if (checkbox) {
    checkbox.disabled = !command.name;
    checkbox.title = !command.name ? 'Command must have a name to add to context menu' : '';
  }
}

export function renderCommandTable(state, actions) {
  const table = $('command-table');
  if (!table) return;
  for (const r of $$('.command-body-row', table)) r.remove();

  for (const command of state.commands) {
    table.appendChild(createCommandRow(command, state, actions));
  }

  updateMoveButtonStates();
  updateOmniboxHighlights(state);
  updateShortcutDisabledStates(state);

  const editor = $('command-editor');
  if (editor) editor.style.display = state.editingCommandId ? 'block' : 'none';
  if (state.editingCommandId) actions.renderEditor();
}

export function updateMoveButtonStates() {
  const rows = $$('.command-body-row', $('command-table'));
  for (const [i, row] of rows.entries()) {
    const moveDeleteGroup = row.querySelector('.move-delete-group');
    if (moveDeleteGroup) {
      moveDeleteGroup.style.display = rows.length <= 1 ? 'none' : '';
    }

    const upBtn = row.querySelector('[title="Move up"]');
    const downBtn = row.querySelector('[title="Move down"]');
    if (upBtn) upBtn.disabled = i === 0;
    if (downBtn) downBtn.disabled = i === rows.length - 1;
  }
}

export function updateOmniboxHighlights(state) {
  const keywords = new Map();
  for (const c of state.commands) {
    const kw = c.triggers?.omnibox;
    if (kw) {
      if (!keywords.has(kw)) keywords.set(kw, []);
      keywords.get(kw).push(c.id);
    }
  }

  const duplicates = new Set();
  for (const ids of keywords.values()) {
    if (ids.length > 1) {
      for (const id of ids) duplicates.add(id);
    }
  }

  for (const row of $$('.command-body-row', $('command-table'))) {
    const id = row.dataset.id;
    const input = row.querySelector('.command-cell-trigger.command-trigger-end input');
    if (input) {
      input.classList.toggle('omnibox-duplicate', duplicates.has(id));
    }
  }
}

export function updateShortcutDisabledStates(state) {
  const usedSlots = new Map();
  for (const command of state.commands) {
    const slot = command.triggers?.shortcutSlot;
    if (slot && !usedSlots.has(slot)) {
      usedSlots.set(slot, command.id);
    }
  }

  for (const row of $$('.command-body-row', $('command-table'))) {
    const id = row.dataset.id;
    const select = row.querySelector('.shortcut-slot-select');
    if (!select) continue;

    const currentValue = select.value;
    for (const option of select.options) {
      if (!option.value) {
        option.disabled = false;
        continue;
      }
      const ownerId = usedSlots.get(option.value);
      option.disabled = Boolean(ownerId && ownerId !== id && option.value !== currentValue);
    }
  }
}
