import { fetchItems } from './engine/item-engine.js';
import { applyFilters, applyRefinement } from './engine/filter-engine.js';
import { executeCommand } from './orchestrators/command-runner.js';
import { updateContextMenus, registerListeners } from './orchestrators/triggers.js';
import { logger } from './engine/logger.js';
import { STORAGE_KEY } from './constants.js';
import { createDefaultState } from './default-state.js';

async function loadState() {
  const result = await chrome.storage.local.get(STORAGE_KEY);
  return result[STORAGE_KEY];
}

async function saveState(state) {
  await chrome.storage.local.set({ [STORAGE_KEY]: state });
}

function hasCommands(state) {
  return !!state && typeof state === 'object' && Array.isArray(state.commands) && state.commands.length > 0;
}

async function ensureState() {
  const state = await loadState();
  if (hasCommands(state)) {
    return state;
  }

  const initialState = createDefaultState();
  await saveState(initialState);
  return initialState;
}

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  logger.info('Message received:', message.type);

  if (message.type === 'RUN_COMMAND') {
    executeCommand(message.command, message.context || {})
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true; // Keep channel open for async response
  }

  if (message.type === 'PREVIEW_ITEMS') {
    (async () => {
      try {
        const items = await fetchItems(message.itemsConfig);
        let filtered = items;

        if (message.filters && message.filters.length > 0) {
          filtered = applyFilters(items, message.filters, message.itemsConfig?.scope);
        }

        if (message.refinement && message.refinement.type !== 'All') {
          filtered = applyRefinement(filtered, message.refinement);
        }

        // Return simplified preview data
        const preview = filtered.slice(0, 10).map(item => ({
          title: item.title || item.name || item.filename || item.url || 'Unknown',
          url: item.url || ''
        }));

        sendResponse({
          success: true,
          count: filtered.length,
          items: preview,
          hasMore: filtered.length > 10
        });
      } catch (error) {
        sendResponse({ success: false, error: error.message });
      }
    })();
    return true;
  }

  if (message.type === 'UPDATE_TRIGGERS') {
    (async () => {
      try {
        const state = await ensureState();
        const commands = Array.isArray(state.commands) ? state.commands : [];
        await updateContextMenus(commands);
        sendResponse({ success: true });
      } catch (error) {
        sendResponse({ success: false, error: error.message });
      }
    })();
    return true;
  }

  if (message.type === 'GET_STATE') {
    ensureState()
      .then(state => sendResponse({ success: true, state }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }

  return false;
});

chrome.storage.onChanged.addListener((changes, areaName) => {
  if (areaName === 'local' && changes[STORAGE_KEY]) {
    const newState = changes[STORAGE_KEY].newValue || { commands: [] };
    const commands = Array.isArray(newState.commands) ? newState.commands : [];
    logger.info('State changed, updating triggers');
    updateContextMenus(commands);
  }
});

chrome.runtime.onInstalled.addListener(async (details) => {
  logger.info(`Extension ${details.reason}: initializing`);

  const state = await ensureState();

  await updateContextMenus(state.commands);

  logger.info('Initialization complete');
});

chrome.runtime.onStartup.addListener(async () => {
  logger.info('Browser startup: initializing triggers');
  const state = await ensureState();
  const commands = Array.isArray(state.commands) ? state.commands : [];
  await updateContextMenus(commands);
});

// Register event listeners at top level.
// These must be registered synchronously so Chrome wakes the worker for these events.
registerListeners();

logger.info('Service worker loaded');
