import { executeCommand } from './command-runner.js';
import { logger } from '../engine/logger.js';
import { STORAGE_KEY } from '../constants.js';

async function loadCommands() {
  const result = await chrome.storage.local.get(STORAGE_KEY);
  const commands = result[STORAGE_KEY]?.commands;
  return Array.isArray(commands) ? commands : [];
}

function findCommand(commands, criteria) {
  if (criteria.iconTrigger) return commands.find((c) => c.triggers?.icon);
  if (criteria.slot) return commands.find((c) => c.triggers?.shortcutSlot === criteria.slot);
  if (criteria.commandId) return commands.find((c) => c.id === criteria.commandId);
  if (criteria.contextMenuId) return commands.find((c) => c.id === criteria.contextMenuId);
  return null;
}

// ---- Toolbar icon ----
function setupIconTrigger() {
  if (chrome.action.onClicked.hasListener(handleIconClick)) {
    chrome.action.onClicked.removeListener(handleIconClick);
  }
  chrome.action.onClicked.addListener(handleIconClick);
}

async function handleIconClick(tab) {
  logger.info('Icon clicked, tab:', tab?.id, tab?.url);
  const commands = await loadCommands();
  const command = findCommand(commands, { iconTrigger: true });
  if (!command) {
    logger.info('No command with icon trigger found, opening options page');
    chrome.runtime.openOptionsPage();
    return;
  }
  logger.info(`Running icon-triggered command: "${command.name}"`);
  await executeCommand(command, { pageUrl: tab?.url || '' });
}

// ---- Keyboard shortcuts ----
function setupShortcutTrigger() {
  if (chrome.commands.onCommand.hasListener(handleShortcutCommand)) {
    chrome.commands.onCommand.removeListener(handleShortcutCommand);
  }
  chrome.commands.onCommand.addListener(handleShortcutCommand);
}

async function handleShortcutCommand(commandId) {
  logger.info('Keyboard shortcut:', commandId);

  const slotMatch = commandId.match(/^run-slot-([a-z])$/);
  if (!slotMatch) {
    logger.warn('Unknown command ID:', commandId);
    return;
  }

  const slot = slotMatch[1].toUpperCase();
  const commands = await loadCommands();
  const command = findCommand(commands, { slot });
  if (!command) {
    logger.warn(`No command found for slot ${slot}`);
    return;
  }

  logger.info(`Running shortcut command (slot ${slot}): "${command.name}"`);
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  await executeCommand(command, { pageUrl: tab?.url || '' });
}

// ---- Context menus ----
let contextClickListenerInstalled = false;

export async function updateContextMenus(commands) {
  await chrome.contextMenus.removeAll();

  const commandsArray = Array.isArray(commands) ? commands : [];
  const contextCommands = commandsArray.filter((c) => c.triggers?.context && c.name);
  for (const command of contextCommands) {
    chrome.contextMenus.create({
      id: command.id,
      title: command.name,
      contexts: ['all']
    }, () => void chrome.runtime.lastError);
  }

  if (!contextClickListenerInstalled) {
    chrome.contextMenus.onClicked.addListener(handleContextClick);
    contextClickListenerInstalled = true;
  }
}

async function handleContextClick(info, tab) {
  logger.info('Context menu clicked:', info.menuItemId);
  const commands = await loadCommands();
  const command = findCommand(commands, { contextMenuId: info.menuItemId });
  if (!command) {
    logger.warn('No command found for context menu ID:', info.menuItemId);
    return;
  }
  logger.info(`Running context menu command: "${command.name}"`);
  await executeCommand(command, {
    triggerText: info.selectionText || '',
    linkUrl: info.linkUrl || '',
    srcUrl: info.srcUrl || '',
    pageUrl: info.pageUrl || tab?.url || ''
  });
}

// ---- Omnibox ----
function setupOmniboxTrigger() {
  chrome.omnibox.onInputChanged.addListener(handleOmniboxSuggest);
  chrome.omnibox.onInputEntered.addListener(handleOmniboxInput);
}

async function handleOmniboxSuggest(text, suggest) {
  const typed = text.trim().toLowerCase();
  const commands = await loadCommands();
  const suggestions = [];
  for (const c of commands) {
    const kw = c.triggers?.omnibox;
    if (!kw) continue;
    if (typed && !kw.toLowerCase().startsWith(typed)) continue;
    suggestions.push({
      content: kw,
      description: `${kw} - ${c.name}`
    });
  }
  suggest(suggestions);
}

async function handleOmniboxInput(text) {
  logger.info('Omnibox input:', text);
  const trimmed = text.trim();
  if (!trimmed) return;

  const spaceIndex = trimmed.indexOf(' ');
  const keyword = spaceIndex === -1 ? trimmed : trimmed.slice(0, spaceIndex);
  const remainder = spaceIndex === -1 ? '' : trimmed.slice(spaceIndex + 1).trim();

  const commands = await loadCommands();
  const command = commands.find((c) => c.triggers?.omnibox === keyword);
  if (!command) {
    logger.warn(`No command found for omnibox keyword: "${keyword}"`);
    return;
  }

  logger.info(`Running omnibox command: "${command.name}" with text: "${remainder}"`);
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  await executeCommand(command, {
    triggerText: remainder,
    pageUrl: tab?.url || ''
  });
}

// ---- Public setup ----

// Register event listeners synchronously at the top level of the service worker.
// In MV3, listeners must be registered during initial script evaluation so Chrome
// knows to wake the service worker for those events.
export function registerListeners() {
  setupIconTrigger();
  setupShortcutTrigger();
  setupOmniboxTrigger();
  logger.info('Top-level event listeners registered');
}
