import { fetchItems } from '../engine/item-engine.js';
import { applyFilters, applyRefinement } from '../engine/filter-engine.js';
import { executeAction, isActionMultiOnly, isActionSingleOnly } from '../engine/action-engine.js';
import { actions as ACTION_DEFS } from '../defs/action-defs.js';
import { logger } from '../engine/logger.js';
import { getIconUrl } from '../utils.js';

function resolveActionLabel(actionId, scope) {
  const scopeActions = ACTION_DEFS[scope] || [];
  const found = scopeActions.find(a => a.id === actionId);
  if (found) return found.label;
  const all = Object.values(ACTION_DEFS).flat();
  const fallback = all.find(a => a.id === actionId);
  return fallback ? fallback.label : actionId;
}

function substituteVariables(text, context) {
  if (!text || typeof text !== 'string') return text;

  const now = new Date();
  const dateStr = now.toISOString().split('T')[0];
  const timeStr = now.toTimeString().split(' ')[0].replace(/:/g, '-');
  const values = {
    date: dateStr,
    time: timeStr,
    domain: context.domain || '',
    url: context.url || '',
    title: context.title || '',
    count: String(context.count || 0),
    index: String(context.index || 0),
    selection: context.selection || '',
    link_url: context.link_url || '',
    src_url: context.src_url || '',
    page_url: context.page_url || ''
  };

  return text.replace(/\\?\{\{([a-zA-Z0-9_]+)\}\}/g, (match, key) => {
    if (match.startsWith('\\')) return `{{${key}}}`;
    if (Object.prototype.hasOwnProperty.call(values, key)) return values[key];
    return match;
  });
}

function showConfirmation(command) {
  return new Promise((resolve) => {
    const notificationId = `tifa-confirm-${Date.now()}`;

    const commandName = String(command?.name || '').trim();
    const message = commandName ? `Run "${commandName}"?` : 'Run?';

    chrome.notifications.create(notificationId, {
      type: 'basic',
      iconUrl: getIconUrl(),
      title: 'Tifa - Confirm Action',
      message: message,
      requireInteraction: true,
      buttons: [{ title: 'Run' }, { title: 'Cancel' }]
    });

    const handleClick = (clickedId) => {
      if (clickedId === notificationId) {
        chrome.notifications.clear(notificationId);
        cleanup();
        resolve(true);
      }
    };

    const handleButton = (clickedId, buttonIndex) => {
      if (clickedId === notificationId) {
        chrome.notifications.clear(notificationId);
        cleanup();
        resolve(buttonIndex === 0);
      }
    };

    const handleClosed = (closedId) => {
      if (closedId === notificationId) {
        cleanup();
        resolve(false);
      }
    };

    const cleanup = () => {
      chrome.notifications.onClicked.removeListener(handleClick);
      chrome.notifications.onButtonClicked.removeListener(handleButton);
      chrome.notifications.onClosed.removeListener(handleClosed);
    };

    chrome.notifications.onClicked.addListener(handleClick);
    chrome.notifications.onButtonClicked.addListener(handleButton);
    chrome.notifications.onClosed.addListener(handleClosed);
  });
}

function safeHostname(url) {
  try {
    if (!url) return '';
    return new URL(url).hostname;
  } catch {
    return '';
  }
}

function processInputs(inputs, context) {
  const processed = {};
  for (const [k, v] of Object.entries(inputs || {})) {
    processed[k] = substituteVariables(v, context);
  }
  return processed;
}

function showNotification(command, itemCount) {
  const scopeLabel = command.items?.scope || 'items';
  const message = itemCount > 0
    ? `${itemCount} ${scopeLabel}${itemCount === 1 ? '' : 's'}`
    : 'Done';

  chrome.notifications.create(`tifa-notify-${Date.now()}`, {
    type: 'basic',
    iconUrl: getIconUrl(),
    title: `Tifa - ${command.name || 'Command'}`,
    message: message
  });
}

export async function executeCommand(command, triggerContext = {}) {
  logger.group(`Executing command: "${command?.name || 'unnamed'}"`);
  try {
    const executionMode = command.executionMode || 'silent';
    logger.info('Execution mode:', executionMode);

    if (executionMode === 'confirm') {
      const confirmed = await showConfirmation(command);
      if (!confirmed) {
        logger.info('User cancelled confirmation');
        logger.groupEnd();
        return { success: false, reason: 'User cancelled' };
      }
    }

    const scope = command.items?.scope || 'tab';
    logger.info('Scope:', scope);

    // Fetch items (global has none)
    const items = await fetchItems(command.items || { scope, selection: '' });
    logger.info(`Fetched ${items.length} items`);

    // Apply filters/refinement (skip for global)
    let selected = items;
    if (scope !== 'global') {
      if (Array.isArray(command.filters) && command.filters.length > 0) {
        selected = applyFilters(items, command.filters, scope);
        logger.info(`After filters: ${selected.length} items (from ${items.length})`);
      }
      if (command.refinement && command.refinement.type && command.refinement.type !== 'All') {
        selected = applyRefinement(selected, command.refinement);
        logger.info(`After refinement: ${selected.length} items`);
      }
    } else {
      selected = [];
    }

    const selectedText = triggerContext.triggerText || triggerContext.selectionText || triggerContext.selection || '';
    const baseContext = {
      count: selected.length,
      selection: selectedText,
      link_url: triggerContext.linkUrl || '',
      src_url: triggerContext.srcUrl || '',
      page_url: triggerContext.pageUrl || ''
    };

    const first = selected[0];
    const itemContext = {
      ...baseContext,
      url: first?.url || '',
      title: first?.title || '',
      domain: safeHostname(first?.url || '')
    };

    const actionList = command.actions || [];
    logger.info(`Running ${actionList.length} action(s)`);

    if (scope !== 'global' && selected.length === 0) {
      logger.warn('No items selected, skipping all actions');
      logger.groupEnd();
      return { success: true };
    }

    for (const action of actionList) {
      const name = action.id || action.name;
      if (!name) continue;

      // Enforce single/multi constraints (when items are involved).
      if (scope !== 'global') {
        const label = resolveActionLabel(name, scope);
        if (isActionSingleOnly(name) && selected.length !== 1) {
          throw new Error(`${label} requires a single item`);
        }
        if (isActionMultiOnly(name) && selected.length < 2) {
          throw new Error(`${label} requires multiple items`);
        }
      }

      const inputs = processInputs(action.inputs || {}, itemContext);

      logger.info(`Executing action: "${name}" on ${selected.length} item(s)`);
      await executeAction(name, scope, selected, inputs, itemContext);
      logger.info(`Action "${name}" completed`);
    }

    if (executionMode === 'notify') {
      showNotification(command, selected.length);
    }

    logger.info('Command completed successfully');
    logger.groupEnd();
    return { success: true };
  } catch (error) {
    logger.warn(`Command failed: ${error.message}`);
    logger.groupEnd();
    if (command.notifyOnError !== false) {
      await logger.notify(`Command failed: ${error.message}`, 'Error');
    }
    return { success: false, error: error.message };
  }
}
