import { items as ITEM_DEFS } from '../defs/item-defs.js';

async function enrichBookmarksWithFolderName(bookmarks) {
  if (!bookmarks || bookmarks.length === 0) return bookmarks;
  try {
    const folderIds = [...new Set(bookmarks.map(b => b.parentId).filter(Boolean))];
    const folderMap = new Map();
    for (const id of folderIds) {
      try {
        const [node] = await chrome.bookmarks.get(id);
        if (node) folderMap.set(id, node.title || '');
      } catch {
        // folder not found
      }
    }
    return bookmarks.map(b => ({
      ...b,
      _folderName: folderMap.get(b.parentId) || ''
    }));
  } catch {
    return bookmarks;
  }
}

async function enrichTabsWithGroupInfo(tabs) {
  if (!tabs || tabs.length === 0) return tabs;
  try {
    const groups = await chrome.tabGroups.query({});
    const groupMap = new Map(groups.map(g => [g.id, g]));
    return tabs.map(t => {
      const group = groupMap.get(t.groupId);
      return {
        ...t,
        _groupTitle: group?.title || '',
        _groupColor: group?.color || ''
      };
    });
  } catch {
    return tabs;
  }
}

export async function fetchItems(itemsConfig) {
  const scope = itemsConfig?.scope || 'tab';
  if (scope === 'global') return [];

  const list = ITEM_DEFS[scope] || [];
  const selectionName = itemsConfig?.selection;
  const def = list.find((d) => d.name === selectionName) || list[0];
  if (!def?.fetch) return [];

  const fetchOpts = {
    includeActiveTab: itemsConfig?.includeActiveTab,
    ...itemsConfig?.options
  };
  let items = await def.fetch(fetchOpts);

  // Enrich items with derived fields for filtering
  if (scope === 'tab') {
    items = await enrichTabsWithGroupInfo(items);
  } else if (scope === 'bookmark') {
    items = await enrichBookmarksWithFolderName(items);
  }

  return items;
}
