import { filters as FILTER_DEFS } from '../defs/filter-defs.js';
import {
  filterOperators,
  applyStringFilter,
  applyNumberFilter,
  applyBooleanFilter,
  applySelectFilter,
  applyDatetimeFilter
} from '../defs/filters-operations.js';
import { shuffleArray } from '../utils.js';

export function applyFilters(items, filters, scope) {
  const list = Array.isArray(items) ? items : [];
  const criteria = Array.isArray(filters) ? filters : [];
  if (criteria.length === 0) return list;

  const defs = scope ? (FILTER_DEFS[scope] || []) : [];

  // Evaluate a single filter criterion against an item
  const evalFilter = (item, c) => {
    const def = defs.find((d) => d.name === c.name);
    const type = normalizeFilterType(c.type, def?.type);
    const operator = c.operator || Object.keys(filterOperators[type] || {})[0] || 'contains';
    const rawValue = def?.getValue ? def.getValue(item) : undefined;
    return matchOne(type, operator, rawValue, c.value);
  };

  // Group filters by OR (AND has higher precedence)
  // Each group contains filters connected by AND
  const groups = [];
  let currentGroup = [];

  for (const [i, c] of criteria.entries()) {
    currentGroup.push(c);

    // If next filter starts with OR, close this group
    const nextFilter = criteria[i + 1];
    if (!nextFilter || nextFilter.join === 'or') {
      groups.push(currentGroup);
      currentGroup = [];
    }
  }

  const testItem = (item) => {
    // Each group must have ALL filters pass (AND within group)
    // At least ONE group must pass (OR between groups)
    return groups.some(group =>
      group.every(c => evalFilter(item, c))
    );
  };

  return list.filter(testItem);
}

export function applyRefinement(items, refinement) {
  const list = Array.isArray(items) ? items : [];
  const type = String(refinement?.type || 'All');
  if (type === 'All') return list;

  const n = Math.max(1, Number(refinement?.n || 1));

  if (type === 'First') return list.slice(0, n);
  if (type === 'Last') return list.slice(-n);
  if (type === 'Random') {
    if (list.length === 0) return [];
    const shuffled = shuffleArray(list);
    return shuffled.slice(0, Math.min(n, shuffled.length));
  }

  return list;
}

function normalizeFilterType(configType, defType) {
  if (Array.isArray(defType)) return 'select';
  if (defType) return defType;
  return configType || 'string';
}

function matchOne(type, operator, itemValue, filterValue) {
  switch (type) {
    case 'string': return applyStringFilter(itemValue, operator, filterValue);
    case 'number': return applyNumberFilter(itemValue, operator, filterValue);
    case 'boolean': return applyBooleanFilter(itemValue, operator, filterValue);
    case 'select': return applySelectFilter(itemValue, operator, filterValue);
    case 'datetime': return applyDatetimeFilter(itemValue, operator, filterValue);
    default: return false;
  }
}
