import { actions as ACTION_DEFS } from '../defs/action-defs.js';
import { logger } from './logger.js';

function findAnyAction(actionName) {
  const all = Object.values(ACTION_DEFS).flat();
  return all.find((a) => a.id === actionName) || all.find((a) => a.label === actionName);
}

function findScopedAction(actionName, scope) {
  const list = ACTION_DEFS[scope] || [];
  return list.find((a) => a.id === actionName) || list.find((a) => a.label === actionName);
}

function normalizeOptionValue(option) {
  if (option && typeof option === 'object' && 'value' in option) {
    return option.value;
  }
  return option;
}

function isInputVisible(inputDef, allInputs) {
  const showWhen = inputDef?.showWhen;
  if (!showWhen || !showWhen.key) return true;
  return allInputs?.[showWhen.key] === showWhen.value;
}

function parseNumericInput(rawValue) {
  if (typeof rawValue === 'number') {
    return Number.isFinite(rawValue) ? rawValue : NaN;
  }
  if (typeof rawValue === 'string') {
    const trimmed = rawValue.trim();
    if (!trimmed) return null;
    const parsed = Number(trimmed);
    return Number.isFinite(parsed) ? parsed : NaN;
  }
  if (rawValue == null) return null;
  return NaN;
}

function validateRequiredInput(inputDef, value) {
  if (!inputDef?.required) return;

  const label = inputDef.label || inputDef.key || 'Input';
  if (inputDef.type === 'checkboxes') {
    if (!Array.isArray(value) || value.length === 0) {
      throw new Error(`${label} is required`);
    }
    return;
  }

  if (inputDef.type === 'number') {
    const parsed = parseNumericInput(value);
    if (parsed === null || Number.isNaN(parsed)) {
      throw new Error(`${label} is required`);
    }
    return;
  }

  if (value == null || String(value).trim() === '') {
    throw new Error(`${label} is required`);
  }
}

function validateInputType(inputDef, value) {
  const label = inputDef.label || inputDef.key || 'Input';
  const type = inputDef.type;

  if (type === 'number') {
    const parsed = parseNumericInput(value);
    if (parsed === null) return;
    if (Number.isNaN(parsed)) {
      throw new Error(`${label} must be a number`);
    }
    if (typeof inputDef.min === 'number' && parsed < inputDef.min) {
      throw new Error(`${label} must be at least ${inputDef.min}`);
    }
    if (typeof inputDef.max === 'number' && parsed > inputDef.max) {
      throw new Error(`${label} must be at most ${inputDef.max}`);
    }
    return;
  }

  if (type === 'select' && Array.isArray(inputDef.options) && value != null && value !== '') {
    const allowed = new Set(inputDef.options.map(normalizeOptionValue));
    if (!allowed.has(value)) {
      throw new Error(`${label} has an invalid value`);
    }
    return;
  }

  if (type === 'checkboxes') {
    if (!Array.isArray(value)) {
      throw new Error(`${label} must be a list`);
    }
    if (Array.isArray(inputDef.options) && inputDef.options.length > 0) {
      const allowed = new Set(inputDef.options.map(normalizeOptionValue));
      const invalid = value.find((entry) => !allowed.has(entry));
      if (invalid !== undefined) {
        throw new Error(`${label} has an invalid value`);
      }
    }
  }
}

function validateZoomStep(value) {
  const normalized = String(value ?? '').trim();
  if (!normalized) return;

  if (normalized.endsWith('%')) {
    const percent = parseFloat(normalized.slice(0, -1));
    if (Number.isNaN(percent) || percent <= 0) {
      throw new Error('Step must be a positive number or percentage');
    }
    return;
  }

  const numeric = parseFloat(normalized);
  if (Number.isNaN(numeric) || numeric <= 0) {
    throw new Error('Step must be a positive number or percentage');
  }
}

function validateActionSpecificInputs(action, inputs) {
  const actionId = action?.id;
  if (actionId === 'move_to_position') {
    const parsed = parseNumericInput(inputs?.index);
    if (parsed === null || Number.isNaN(parsed)) {
      throw new Error('Index must be a valid number');
    }
  }

  if (actionId === 'zoom_in' || actionId === 'zoom_out') {
    validateZoomStep(inputs?.step);
  }
}

function validateActionInputs(action, inputs = {}) {
  const inputDefs = Array.isArray(action?.inputs) ? action.inputs : [];

  for (const inputDef of inputDefs) {
    if (!isInputVisible(inputDef, inputs)) continue;
    const value = inputs[inputDef.key];
    validateRequiredInput(inputDef, value);
    validateInputType(inputDef, value);
  }

  validateActionSpecificInputs(action, inputs);
}

export async function executeAction(actionName, scope, items, inputs = {}, context = {}) {
  const action = findScopedAction(actionName, scope);
  if (!action) {
    logger.error(`Unknown action: ${actionName} (scope: ${scope})`);
    throw new Error(`Unknown action: ${actionName} (scope: ${scope})`);
  }

  validateActionInputs(action, inputs);

  const list = Array.isArray(items) ? items : [];
  const selection = action.selection || (scope === 'global' ? 'none' : 'each');
  logger.debug(`Action "${actionName}" selection mode: ${selection}, items: ${list.length}`);

  if (selection === 'none') {
    return await action.execute(inputs, context);
  }

  if (selection === 'group') {
    return await action.execute(list, inputs, context);
  }

  if (selection === 'single') {
    return await action.execute(list[0], inputs, context);
  }

  const failures = [];
  const chromeUrlFailures = [];
  for (const item of list) {
    try {
      await action.execute(item, inputs, context);
    } catch (error) {
      const message = error?.message || String(error);
      const url = item?.url || '';
      const isChromeUrl = url && !/^https?:\/\//i.test(url);
      if (isChromeUrl) {
        chromeUrlFailures.push(message);
        logger.debug(`Action "${actionName}" skipped on internal URL: ${url}`);
      } else {
        failures.push(message);
        logger.warn(`Action "${actionName}" failed for one item`, { error: message });
      }
    }
  }

  if (failures.length > 0) {
    const label = action.label || actionName;
    const sample = failures.slice(0, 3).join('; ');
    const suffix = failures.length > 3 ? `; +${failures.length - 3} more` : '';
    const countPart = list.length <= 1 ? ''
      : failures.length === list.length ? ` on ${list.length} items`
      : ` on ${failures.length}/${list.length} items`;
    throw new Error(`${label} failed${countPart}: ${sample}${suffix}`);
  }

  if (chromeUrlFailures.length > 0 && chromeUrlFailures.length === list.length) {
    const label = action.label || actionName;
    const itemWord = list.length === 1 ? 'tab' : 'tabs';
    throw new Error(`${label} can't run on internal browser ${itemWord}`);
  }
}

export function isActionSingleOnly(actionName) {
  const a = findAnyAction(actionName);
  return a?.selection === 'single';
}

export function isActionMultiOnly(actionName) {
  const a = findAnyAction(actionName);
  if (a?.selection !== 'group') return false;
  // Actions with minItems: 1 can work on a single item
  return (a.minItems ?? 2) >= 2;
}
