const flattenBookmarks = (nodes, result = []) => {
  for (const node of nodes) {
    if (node.url) {
      result.push(node);
    }
    if (node.children) {
      flattenBookmarks(node.children, result);
    }
  }
  return result;
};

const findFolderByName = (nodes, name) => {
  for (const node of nodes) {
    if (!node.url && node.title.toLowerCase() === name.toLowerCase()) {
      return node;
    }
    if (node.children) {
      const found = findFolderByName(node.children, name);
      if (found) return found;
    }
  }
  return null;
};

// "Foldered" / keyed structure by scope
export const items = {
  window: [
    {
      type: 'item',
      name: 'Focused Window',
      fetch: async () => {
        const win = await chrome.windows.getLastFocused({ populate: true });
        return win ? [win] : [];
      }
    },
    {
      type: 'item',
      name: 'All Windows',
      fetch: async () => {
        return await chrome.windows.getAll({ populate: true });
      }
    },
    {
      type: 'item',
      name: 'All Except Focused',
      fetch: async () => {
        const windows = await chrome.windows.getAll({ populate: true });
        const focused = await chrome.windows.getLastFocused();
        return windows.filter(w => w.id !== focused.id);
      }
    },
    {
      type: 'item',
      name: 'Recently Closed Windows',
      fetch: async () => {
        const sessions = await chrome.sessions.getRecentlyClosed({ maxResults: 25 });
        return sessions
          .filter(s => s.window)
          .map(s => ({
            ...s.window,
            sessionId: s.window.sessionId,
            isRecentlyClosed: true
          }));
      }
    }
  ],

  bookmark: [
    {
      type: 'item',
      name: 'All Bookmarks',
      fetch: async () => {
        const tree = await chrome.bookmarks.getTree();
        return flattenBookmarks(tree);
      }
    },
    {
      type: 'item',
      name: 'In Folder',
      fetch: async (folderName) => {
        if (!folderName) return [];
        const tree = await chrome.bookmarks.getTree();
        const folder = findFolderByName(tree, folderName);
        if (!folder) return [];
        return flattenBookmarks(folder.children || []);
      }
    }
  ],

  download: [
    {
      type: 'item',
      name: 'Recent',
      options: [
        { key: 'limit', type: 'number', label: 'Max', default: 100, min: 0, step: 1 }
      ],
      fetch: async (opts) => {
        const query = { orderBy: ['-startTime'] };
        const parsedLimit = parseInt(opts?.limit, 10);
        const limit = Number.isFinite(parsedLimit) ? parsedLimit : 100;
        if (limit > 0) query.limit = limit;
        return await chrome.downloads.search(query);
      }
    },
    {
      type: 'item',
      name: 'Active / In Progress',
      fetch: async () => {
        return await chrome.downloads.search({
          state: 'in_progress'
        });
      }
    }
  ],

  history: [
    {
      type: 'item',
      name: 'Last 24 Hours',
      fetch: async () => {
        const DAY_MS = 24 * 60 * 60 * 1000;
        return await chrome.history.search({
          text: '',
          startTime: Date.now() - DAY_MS,
          maxResults: 1000
        });
      }
    },
    {
      type: 'item',
      name: 'Last 7 Days',
      fetch: async () => {
        const DAY_MS = 24 * 60 * 60 * 1000;
        return await chrome.history.search({
          text: '',
          startTime: Date.now() - (7 * DAY_MS),
          maxResults: 1000
        });
      }
    },
    {
      type: 'item',
      name: 'All Time',
      fetch: async () => {
        return await chrome.history.search({
          text: '',
          startTime: 0,
          maxResults: 10000
        });
      }
    }
  ],

  reading_list: [
    {
      type: 'item',
      name: 'All Items',
      fetch: async () => {
        if (!chrome.readingList) return [];
        return await chrome.readingList.query({});
      }
    },
    {
      type: 'item',
      name: 'Unread Only',
      fetch: async () => {
        if (!chrome.readingList) return [];
        return await chrome.readingList.query({ hasBeenRead: false });
      }
    },
    {
      type: 'item',
      name: 'Read Only',
      fetch: async () => {
        if (!chrome.readingList) return [];
        return await chrome.readingList.query({ hasBeenRead: true });
      }
    }
  ],

  tab: [
    {
      type: 'item',
      name: 'Active Tab',
      fetch: async () => {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        return tab ? [tab] : [];
      }
    },
    {
      type: 'item',
      name: 'In Current Window',
      fetch: async (options = {}) => {
        const includeActiveTab = options.includeActiveTab !== false;
        const tabs = await chrome.tabs.query({ currentWindow: true });
        if (!includeActiveTab) {
          const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
          return tabs.filter(t => t.id !== activeTab?.id);
        }
        return tabs;
      }
    },
    {
      type: 'item',
      name: 'In All Windows',
      fetch: async (options = {}) => {
        const includeActiveTab = options.includeActiveTab !== false;
        const tabs = await chrome.tabs.query({});
        if (!includeActiveTab) {
          const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
          return tabs.filter(t => t.id !== activeTab?.id);
        }
        return tabs;
      }
    },
    {
      type: 'item',
      name: 'Left of Active Tab',
      fetch: async (options = {}) => {
        const includeActiveTab = options.includeActiveTab !== false;
        const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!activeTab) return [];
        const tabs = await chrome.tabs.query({ currentWindow: true });
        let leftTabs = tabs.filter(t => t.index < activeTab.index);
        if (includeActiveTab) leftTabs.push(activeTab);
        return leftTabs;
      }
    },
    {
      type: 'item',
      name: 'Right of Active Tab',
      fetch: async (options = {}) => {
        const includeActiveTab = options.includeActiveTab !== false;
        const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!activeTab) return [];
        const tabs = await chrome.tabs.query({ currentWindow: true });
        let rightTabs = tabs.filter(t => t.index > activeTab.index);
        if (includeActiveTab) rightTabs.unshift(activeTab);
        return rightTabs;
      }
    },
    {
      type: 'item',
      name: 'Highlighted',
      fetch: async (options = {}) => {
        const includeActiveTab = options.includeActiveTab !== false;
        const tabs = await chrome.tabs.query({ highlighted: true, currentWindow: true });
        if (!includeActiveTab) {
          const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
          return tabs.filter(t => t.id !== activeTab?.id);
        }
        return tabs;
      }
    },
    {
      type: 'item',
      name: 'Recently Closed Tabs',
      fetch: async () => {
        const sessions = await chrome.sessions.getRecentlyClosed({ maxResults: 25 });
        return sessions
          .filter(s => s.tab)
          .map(s => ({
            ...s.tab,
            sessionId: s.tab.sessionId,
            isRecentlyClosed: true
          }));
      }
    }
  ],

 
  group: [
    {
      type: 'item',
      name: 'All Tab Groups',
      fetch: async () => {
        return await chrome.tabGroups.query({});
      }
    },
    {
      type: 'item',
      name: 'Containing Active Tab',
      fetch: async () => {
        const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!activeTab || activeTab.groupId === undefined || activeTab.groupId < 0) return [];
        const group = await chrome.tabGroups.get(activeTab.groupId);
        return group ? [group] : [];
      }
    },
    {
      type: 'item',
      name: 'In Current Window',
      fetch: async () => {
        const win = await chrome.windows.getLastFocused();
        return await chrome.tabGroups.query({ windowId: win.id });
      }
    }
  ]
};
