export const filterOperators = {
  string: {
    "Contains": (a, b) => a.includes(b),
    "Is exactly": (a, b) => a === b,
    "Is not exactly": (a, b) => a !== b,
    "Starts with": (a, b) => a.startsWith(b),
    "Doesn't start with": (a, b) => !a.startsWith(b),
    "Ends with": (a, b) => a.endsWith(b),
    "Doesn't end with": (a, b) => !a.endsWith(b),
    "Regex": (a, b) => {
      try {
        return new RegExp(b, "i").test(a);
      } catch {
        return false;
      }
    },
    "Doesn't match regex": (a, b) => {
      try {
        return !new RegExp(b, "i").test(a);
      } catch {
        return false;
      }
    },
    "Doesn't contain": (a, b) => !a.includes(b)
  },
  number: {
    "Equals": (a, b) => a === b,
    "Greater than": (a, b) => a > b,
    "Less than": (a, b) => a < b
  },
  boolean: {
    "Is": (a, b) => a === (b === "true" || b === true)
  },
  select: {
    "Is": (a, b) => a === b
  },
  datetime: {
    "Before": (a, b) => a < b,
    "After": (a, b) => a > b
  }
};

const normalize = {
  string: (val) => String(val).toLowerCase(),
  number: (val) => Number(val),
  boolean: (val) => Boolean(val),
  datetime: (val) => Number(val)
};

const normalizeFilter = {
  string: (val) => String(val).toLowerCase(),
  number: (val) => Number(val),
  boolean: (val) => val, // Already handled in operator comparison or pre-normalized
  datetime: (val) => new Date(val).getTime()
};

const applyFilter = (type, value, operator, filterValue) => {
  if (value === undefined || value === null) return false;

  const ops = filterOperators[type];
  if (!ops) return false;
  const operatorAliases = {
    string: {
      "Equals": "Is exactly"
    }
  };
  const resolvedOperator = operatorAliases[type]?.[operator] || operator;
  if (!ops[resolvedOperator]) return false;

  const normVal = normalize[type] ? normalize[type](value) : value;
  const normFilter = normalizeFilter[type] ? normalizeFilter[type](filterValue) : filterValue;

  if (type === "number" || type === "datetime") {
    if (isNaN(normVal) || isNaN(normFilter)) return false;
  }

  return ops[resolvedOperator](normVal, normFilter);
};

export const applyStringFilter = (v, op, f) => applyFilter("string", v, op, f);
export const applyNumberFilter = (v, op, f) => applyFilter("number", v, op, f);
export const applyBooleanFilter = (v, op, f) => applyFilter("boolean", v, op, f);
export const applySelectFilter = (v, op, f) => applyFilter("select", v, op, f);
export const applyDatetimeFilter = (v, op, f) => applyFilter("datetime", v, op, f);
