import { logger } from '../engine/logger.js';
import { shuffleArray } from '../utils.js';

export const ZOOM_LIMITS = { min: 0.25, max: 5.0 };

const SITE_CONTENT_SETTING_OPTIONS = [
  { label: "Images", value: "images" },
  { label: "JavaScript", value: "javascript" },
  { label: "Notifications", value: "notifications" },
  { label: "Location", value: "location" },
  { label: "Camera", value: "camera" },
  { label: "Microphone", value: "microphone" },
  { label: "Automatic Downloads", value: "automaticDownloads" },
  { label: "Cookies", value: "cookies" },
];

const SITE_CONTENT_SETTING_DEFAULTS = SITE_CONTENT_SETTING_OPTIONS.map((option) => option.value);

const GLOBAL_CONTENT_SETTING_OPTIONS = [
  { label: "Popups", value: "popups" },
  { label: "Images", value: "images" },
  { label: "JavaScript", value: "javascript" },
  { label: "Notifications", value: "notifications" },
  { label: "Location", value: "location" },
  { label: "Camera", value: "camera" },
  { label: "Microphone", value: "microphone" },
  { label: "Automatic Downloads", value: "automaticDownloads" },
  { label: "Cookies", value: "cookies" },
  { label: "Sensors", value: "sensors" },
];

const GLOBAL_CONTENT_SETTING_DEFAULTS = GLOBAL_CONTENT_SETTING_OPTIONS.map((option) => option.value);

const TAB_GROUP_COLOR_OPTIONS_WITH_UNSPECIFIED = [
  { label: "Unspecified", value: "unspecified" },
  { label: "Grey", value: "grey" },
  { label: "Blue", value: "blue" },
  { label: "Red", value: "red" },
  { label: "Yellow", value: "yellow" },
  { label: "Green", value: "green" },
  { label: "Pink", value: "pink" },
  { label: "Purple", value: "purple" },
  { label: "Cyan", value: "cyan" },
  { label: "Orange", value: "orange" },
];

const TAB_GROUP_COLOR_OPTIONS = TAB_GROUP_COLOR_OPTIONS_WITH_UNSPECIFIED.slice(1);

export const actions = {
  bookmark: [
    // --- Opening ---
    {
      id: "open_new_tab",
      label: "Open in New Tab",
      selection: "each",
      execute: (bookmark) => chrome.tabs.create({ url: bookmark.url })
    },
    {
      id: "open_background",
      label: "Open in Background",
      selection: "each",
      execute: (bookmark) => chrome.tabs.create({ url: bookmark.url, active: false })
    },
    {
      id: "open_new_window",
      label: "Open in New Window",
      selection: "each",
      execute: (bookmark) => chrome.windows.create({ url: bookmark.url })
    },
    {
      id: "open_incognito",
      label: "Open in Incognito Window",
      selection: "each",
      execute: (bookmark) => chrome.windows.create({ url: bookmark.url, incognito: true })
    },
    {
      id: "open_in_new_group",
      label: "Open in New Group",
      selection: "group",
      minItems: 1,
      inputs: [
        { key: "color", type: "select", label: "Color", options: TAB_GROUP_COLOR_OPTIONS_WITH_UNSPECIFIED },
        { key: "name", type: "text", label: "Name" },
      ],
      execute: async (bookmarks, inputs) => {
        const tabIds = [];
        for (const bookmark of bookmarks) {
          if (!bookmark.url) continue;
          const tab = await chrome.tabs.create({ url: bookmark.url, active: false });
          tabIds.push(tab.id);
        }
        if (tabIds.length === 0) return;
        const groupId = await chrome.tabs.group({ tabIds });
        const color = (inputs.color || '').toLowerCase();
        const updateProps = {};
        if (inputs.name) updateProps.title = inputs.name;
        if (color && color !== 'unspecified') updateProps.color = color;
        if (Object.keys(updateProps).length > 0) {
          await chrome.tabGroups.update(groupId, updateProps);
        }
      }
    },

    // --- Duplication ---
    {
      id: "duplicate",
      label: "Duplicate",
      selection: "each",
      execute: async (bookmark) => {
        await chrome.bookmarks.create({
          parentId: bookmark.parentId,
          title: bookmark.title,
          url: bookmark.url,
          index: bookmark.index + 1
        });
      }
    },

    // --- Editing ---
    {
      id: "rename",
      label: "Rename",
      selection: "each",
      input: "text",
      inputs: [
        { key: "title", type: "text", label: "Title", required: true },
      ],
      execute: async (bookmark, inputs) => {
        await chrome.bookmarks.update(bookmark.id, { title: inputs.title });
      }
    },
    {
      id: "edit_url",
      label: "Edit URL",
      selection: "each",
      input: "text",
      inputs: [
        { key: "url", type: "text", label: "URL", required: true },
      ],
      execute: async (bookmark, inputs) => {
        await chrome.bookmarks.update(bookmark.id, { url: inputs.url });
      }
    },

    // --- Folder Management ---
    {
      id: "create_folder",
      label: "Create Folder",
      selection: "none",
      input: "text",
      inputs: [
        { key: "folder", type: "text", label: "Name", required: true },
        { key: "destination", type: "select", label: "Create under", options: ["Bookmarks Bar", "Other Bookmarks", "Custom"] },
        { key: "parentFolder", type: "text", label: "Parent folder", required: true, showWhen: { key: "destination", value: "Custom" } },
      ],
      execute: async (bookmark, inputs) => {
        const parentId = await resolveBookmarkParentFolder(inputs);
        await chrome.bookmarks.create({ title: inputs.folder, parentId });
      }
    },

    // --- Organization / Movement ---
    {
      id: "move_to_folder",
      label: "Move to Folder",
      selection: "each",
      input: "text",
      inputs: [
        { key: "destination", type: "select", label: "Destination", options: ["Bookmarks Bar", "Other Bookmarks", "Custom"] },
        { key: "folder", type: "text", label: "Folder", required: true, showWhen: { key: "destination", value: "Custom" } },
      ],
      execute: async (bookmark, inputs) => {
        const parentId = await resolveBookmarkFolder(inputs);
        await chrome.bookmarks.move(bookmark.id, { parentId });
      }
    },
    {
      id: "sort_by_url",
      label: "Sort by URL",
      selection: "group",
      execute: async (bookmarks) => {
        const sorted = [...bookmarks].sort((a, b) => (a.url || "").localeCompare(b.url || ""));
        for (const [i, bookmark] of sorted.entries()) {
          await chrome.bookmarks.move(bookmark.id, { index: i });
        }
      }
    },
    {
      id: "sort_by_title",
      label: "Sort by Title",
      selection: "group",
      execute: async (bookmarks) => {
        const sorted = [...bookmarks].sort((a, b) => (a.title || "").localeCompare(b.title || ""));
        for (const [i, bookmark] of sorted.entries()) {
          await chrome.bookmarks.move(bookmark.id, { index: i });
        }
      }
    },
    {
      id: "shuffle",
      label: "Shuffle",
      selection: "group",
      execute: async (bookmarks) => {
        const shuffled = shuffleArray(bookmarks);
        for (const [i, bookmark] of shuffled.entries()) {
          await chrome.bookmarks.move(bookmark.id, { index: i });
        }
      }
    },

    // --- Deduplication ---
    {
      id: "delete_duplicate_bookmarks_keep_first",
      label: "Delete Duplicate Bookmarks (Keep First)",
      selection: "group",
      execute: async (bookmarks) => {
        const seen = new Set();
        for (const bookmark of bookmarks) {
          if (!bookmark.url) continue;
          if (seen.has(bookmark.url)) {
            await chrome.bookmarks.remove(bookmark.id);
          } else {
            seen.add(bookmark.url);
          }
        }
      }
    },
    {
      id: "delete_duplicate_bookmarks_keep_last",
      label: "Delete Duplicate Bookmarks (Keep Last)",
      selection: "group",
      execute: async (bookmarks) => {
        const lastByUrl = new Map();
        for (const bookmark of bookmarks) {
          if (!bookmark.url) continue;
          lastByUrl.set(bookmark.url, bookmark.id);
        }
        const keepIds = new Set(lastByUrl.values());
        for (const bookmark of bookmarks) {
          if (bookmark.url && !keepIds.has(bookmark.id)) {
            await chrome.bookmarks.remove(bookmark.id);
          }
        }
      }
    },

    // --- URL Manipulation ---
    {
      id: "find_replace_in_url",
      label: "Find & Replace in URL",
      selection: "each",
      input: "text",
      inputs: [
        { key: "find", type: "text", label: "Find (Regex)", required: true },
        { key: "replace", type: "text", label: "Replace" },
      ],
      execute: async (bookmark, inputs) => {
        const replace = inputs.replace || "";
        const regex = new RegExp(inputs.find, "g");
        if (!bookmark.url) return;
        const newUrl = bookmark.url.replace(regex, replace);
        if (newUrl !== bookmark.url) {
          await chrome.bookmarks.update(bookmark.id, { url: newUrl });
        }
      }
    },

    {
      id: "find_replace_in_title",
      label: "Find & Replace in Title",
      selection: "each",
      input: "text",
      inputs: [
        { key: "find", type: "text", label: "Find (Regex)", required: true },
        { key: "replace", type: "text", label: "Replace" },
      ],
      execute: async (bookmark, inputs) => {
        const replace = inputs.replace || "";
        const regex = new RegExp(inputs.find, "g");
        const currentTitle = bookmark.title || "";
        const newTitle = currentTitle.replace(regex, replace);
        if (newTitle !== currentTitle) {
          await chrome.bookmarks.update(bookmark.id, { title: newTitle });
        }
      }
    },

    // --- Export / Clipboard ---
    {
      id: "copy_urls_to_clipboard",
      label: "Copy URLs to Clipboard",
      selection: "group",
      minItems: 1,
      execute: async (bookmarks) => {
        const text = (bookmarks || []).map(b => b.url).filter(Boolean).join("\n");
        await copyToClipboard(text);
      }
    },
    {
      id: "download_bookmarks_json",
      label: "Download Bookmarks (JSON)",
      selection: "group",
      minItems: 1,
      execute: async (bookmarks) => {
        const json = JSON.stringify(bookmarks || [], null, 2);
        const blob = new Blob([json], { type: "application/json" });
        const url = URL.createObjectURL(blob);
        await chrome.downloads.download({ url, filename: "bookmarks-export.json" });
      }
    },

    // --- Deletion ---
    {
      id: "delete",
      label: "Delete",
      selection: "each",
      execute: (bookmark) => chrome.bookmarks.remove(bookmark.id)
    }
  ],

  download: [
    // --- Open / View ---
    {
      id: "open",
      label: "Open",
      selection: "each",
      execute: (download) => chrome.downloads.open(download.id)
    },
    {
      id: "show_in_folder",
      label: "Show in Folder",
      selection: "each",
      execute: (download) => chrome.downloads.show(download.id)
    },

    // --- State Control ---
    {
      id: "pause",
      label: "Pause",
      selection: "each",
      execute: (download) => chrome.downloads.pause(download.id)
    },
    {
      id: "resume",
      label: "Resume",
      selection: "each",
      execute: (download) => chrome.downloads.resume(download.id)
    },
    {
      id: "toggle_pause",
      label: "Toggle Pause",
      selection: "each",
      execute: async (download) => {
        if (download.paused) {
          await chrome.downloads.resume(download.id);
        } else {
          await chrome.downloads.pause(download.id);
        }
      }
    },
    {
      id: "cancel",
      label: "Cancel",
      selection: "each",
      execute: (download) => chrome.downloads.cancel(download.id)
    },

    // --- Removal ---
    {
      id: "remove_from_history",
      label: "Remove from History",
      selection: "each",
      execute: (download) => chrome.downloads.erase({ id: download.id })
    },
    {
      id: "delete_file",
      label: "Delete File",
      selection: "each",
      execute: (download) => chrome.downloads.removeFile(download.id)
    }
  ],

  history: [
    // --- Opening ---
    {
      id: "open_new_tab",
      label: "Open in New Tab",
      selection: "each",
      execute: (visit) => chrome.tabs.create({ url: visit.url })
    },
    {
      id: "open_background",
      label: "Open in Background",
      selection: "each",
      execute: (visit) => chrome.tabs.create({ url: visit.url, active: false })
    },
    {
      id: "open_new_window",
      label: "Open in New Window",
      selection: "each",
      execute: (visit) => chrome.windows.create({ url: visit.url })
    },
    {
      id: "open_incognito",
      label: "Open in Incognito Window",
      selection: "each",
      execute: (visit) => chrome.windows.create({ url: visit.url, incognito: true })
    },
    {
      id: "open_in_new_group",
      label: "Open in New Group",
      selection: "group",
      minItems: 1,
      inputs: [
        { key: "color", type: "select", label: "Color", options: TAB_GROUP_COLOR_OPTIONS_WITH_UNSPECIFIED },
        { key: "name", type: "text", label: "Name" },
      ],
      execute: async (visits, inputs) => {
        const tabIds = [];
        for (const visit of visits) {
          if (!visit.url) continue;
          const tab = await chrome.tabs.create({ url: visit.url, active: false });
          tabIds.push(tab.id);
        }
        if (tabIds.length === 0) return;
        const groupId = await chrome.tabs.group({ tabIds });
        const color = (inputs.color || '').toLowerCase();
        const updateProps = {};
        if (inputs.name) updateProps.title = inputs.name;
        if (color && color !== 'unspecified') updateProps.color = color;
        if (Object.keys(updateProps).length > 0) {
          await chrome.tabGroups.update(groupId, updateProps);
        }
      }
    },

    // --- Export / Clipboard ---
    {
      id: "copy_url_to_clipboard",
      label: "Copy URL to Clipboard",
      selection: "each",
      execute: async (visit) => {
        await copyToClipboard(visit.url);
      }
    },

    // --- Save / Convert ---
    {
      id: "add_to_bookmarks",
      label: "Add to Bookmarks",
      selection: "each",
      input: "text",
      inputs: [
        { key: "destination", type: "select", label: "Destination", options: ["(Default)", "Bookmarks Bar", "Other Bookmarks", "Custom"] },
        { key: "folder", type: "text", label: "Folder", required: true, showWhen: { key: "destination", value: "Custom" } },
      ],
      execute: async (visit, inputs) => {
        let parentId;
        if (inputs.destination && inputs.destination !== '(Default)') {
          parentId = await resolveBookmarkFolder(inputs);
        }
        await chrome.bookmarks.create({
          parentId,
          title: visit.title || visit.url,
          url: visit.url
        });
      }
    },
    {
      id: "add_to_reading_list",
      label: "Add to Reading List",
      selection: "each",
      execute: async (visit) => {
        await chrome.readingList.addEntry({
          url: visit.url,
          title: visit.title || visit.url,
          hasBeenRead: false
        });
      }
    },

    // --- Deletion ---
    {
      id: "delete_url",
      label: "Delete from History",
      selection: "each",
      execute: (visit) => chrome.history.deleteUrl({ url: visit.url })
    }
  ],

  global: [
    // --- Search & Navigation ---
    {
      id: "search",
      label: "Search",
      input: "text",
      inputs: [
        { key: "query", type: "text", label: "Query", required: true },
        { key: "disposition", type: "select", label: "Open in", options: ["New Tab", "Current Tab", "New Window"] },
      ],
      execute: (inputs) => {
        const dispositionMap = { 'Current Tab': 'CURRENT_TAB', 'New Window': 'NEW_WINDOW' };
        const disposition = dispositionMap[inputs.disposition] || 'NEW_TAB';
        return chrome.search.query({ text: inputs.query, disposition });
      }
    },
    {
      id: "restore_recently_closed",
      label: "Restore Recently Closed",
      execute: async () => {
        const sessions = await chrome.sessions.getRecentlyClosed({ maxResults: 1 });
        if (sessions.length === 0) return;
        const session = sessions[0];
        const sessionId = session.tab?.sessionId || session.window?.sessionId;
        if (!sessionId) return;
        await chrome.sessions.restore(sessionId);
      }
    },

    // --- Windows & Panels ---
    {
      id: "create_new_window",
      label: "Create New Window",
      execute: () => chrome.windows.create({})
    },
    {
      id: "create_incognito_window",
      label: "Create Incognito Window",
      execute: () => chrome.windows.create({ incognito: true })
    },
    {
      id: "open_downloads_folder",
      label: "Open Downloads Folder",
      execute: () => chrome.downloads.showDefaultFolder()
    },
    {
      id: "open_tifa_options",
      label: "Open Tifa Options",
      execute: () => chrome.runtime.openOptionsPage()
    },
    {
      id: "copy_text_to_clipboard",
      label: "Copy Text to Clipboard",
      input: "textarea",
      inputs: [
        { key: "mode", type: "select", label: "Mode", options: ["Plain text", "HTML"], default: "Plain text" },
        { key: "text", type: "textarea", label: "Text", required: true },
      ],
      execute: async (inputs) => {
        await copyToClipboard(inputs.text, inputs.mode);
      }
    },
    {
      id: "add_to_bookmarks",
      label: "Add to Bookmarks",
      selection: "none",
      input: "text",
      inputs: [
        { key: "url", type: "text", label: "URL", required: true },
        { key: "title", type: "text", label: "Name" },
        { key: "destination", type: "select", label: "Destination", options: ["(Default)", "Bookmarks Bar", "Other Bookmarks", "Custom"] },
        { key: "folder", type: "text", label: "Folder", required: true, showWhen: { key: "destination", value: "Custom" } },
      ],
      execute: async (inputs) => {
        const url = (inputs.url || "").trim();
        let parentId;
        if (inputs.destination && inputs.destination !== "(Default)") {
          parentId = await resolveBookmarkFolder(inputs);
        }
        await chrome.bookmarks.create({
          parentId,
          title: (inputs.title || "").trim() || url,
          url
        });
      }
    },
    {
      id: "insert_text",
      label: "Insert Text",
      input: "textarea",
      inputs: [
        { key: "text", type: "textarea", label: "Text", required: true },
      ],
      execute: async (inputs) => {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) throw new Error("No active tab");
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (value) => {
            document.execCommand('insertText', false, value);
          },
          args: [inputs.text]
        });
      }
    },

    // --- System Info / Export ---
    {
      id: "get_info_copy_to_clipboard",
      label: "Get Info (Copy to Clipboard)",
      inputs: [
        { key: "types", type: "checkboxes", label: "Info types", options: [
          { label: "CPU", value: "cpu" },
          { label: "Memory", value: "memory" },
          { label: "Storage", value: "storage" },
          { label: "Display", value: "display" },
          { label: "Extensions", value: "extensions" },
          { label: "Top Sites", value: "topSites" },
          { label: "Fonts", value: "fonts" },
          { label: "Tabs", value: "tabs" },
        ], default: ["cpu", "memory", "storage", "display", "extensions", "topSites", "fonts", "tabs"] },
      ],
      execute: async (inputs) => {
        const types = Array.isArray(inputs?.types) && inputs.types.length > 0 ? inputs.types : null;
        const info = await getSystemInfo(types);
        await copyToClipboard(JSON.stringify(info, null, 2));
      }
    },
    {
      id: "get_info_download_json",
      label: "Get Info (Download JSON)",
      inputs: [
        { key: "types", type: "checkboxes", label: "Info types", options: [
          { label: "CPU", value: "cpu" },
          { label: "Memory", value: "memory" },
          { label: "Storage", value: "storage" },
          { label: "Display", value: "display" },
          { label: "Extensions", value: "extensions" },
          { label: "Top Sites", value: "topSites" },
          { label: "Fonts", value: "fonts" },
          { label: "Tabs", value: "tabs" },
        ], default: ["cpu", "memory", "storage", "display", "extensions", "topSites", "fonts", "tabs"] },
      ],
      execute: async (inputs) => {
        const types = Array.isArray(inputs?.types) && inputs.types.length > 0 ? inputs.types : null;
        const info = await getSystemInfo(types);
        const filename = `system-info-${new Date().getTime()}.json`;
        const json = JSON.stringify(info, null, 2);
        const blob = new Blob([json], { type: "application/json" });
        const url = URL.createObjectURL(blob);
        await chrome.downloads.download({ url, filename });
      }
    },
    {
      id: "open_all_top_sites",
      label: "Open All Top Sites",
      inputs: [
        { key: "limit", type: "number", label: "Limit", default: 10, min: 1, max: 20, step: 1 },
      ],
      execute: async (inputs) => {
        const parsedLimit = parseInt(inputs.limit, 10);
        const limit = !isNaN(parsedLimit) ? Math.min(20, Math.max(1, parsedLimit)) : 10;
        const sites = await chrome.topSites.get();
        for (const site of sites.slice(0, limit)) {
          await chrome.tabs.create({ url: site.url, active: false });
        }
      }
    },
    {
      id: "set_default_font_size",
      label: "Set Default Font Size",
      inputs: [
        { key: "size", type: "number", label: "Size", default: 16, min: 6, max: 72, step: 1 },
      ],
      execute: async (inputs) => {
        const size = parseInt(inputs.size, 10);
        await chrome.fontSettings.setDefaultFontSize({ pixelSize: size });
      }
    },
    {
      id: "set_minimum_font_size",
      label: "Set Minimum Font Size",
      inputs: [
        { key: "size", type: "number", label: "Size", default: 0, min: 0, max: 24, step: 1 },
      ],
      execute: async (inputs) => {
        const size = parseInt(inputs.size, 10);
        await chrome.fontSettings.setMinimumFontSize({ pixelSize: size });
      }
    },

    // --- Power & Extension Lifecycle ---
    {
      id: "keep_awake_display",
      label: "Keep Awake (Display)",
      execute: () => {
        chrome.power.requestKeepAwake('display');
      }
    },
    {
      id: "keep_awake_system",
      label: "Keep Awake (System)",
      execute: () => {
        chrome.power.requestKeepAwake("system");
      }
    },
    {
      id: "allow_sleep",
      label: "Allow Sleep",
      execute: () => {
        chrome.power.releaseKeepAwake();
      }
    },
    {
      id: "hard_reload_extension",
      label: "Hard Reload Extension",
      execute: () => chrome.runtime.reload()
    },

    // --- Browsing Data ---
    {
      id: "clear_browsing_data_all_sites",
      label: "Clear Browsing Data (All Sites)",
      inputs: [
        { key: "types", type: "checkboxes", label: "Data types", options: [
          { label: "Cache", value: "cache" },
          { label: "Cookies", value: "cookies" },
          { label: "History", value: "history" },
          { label: "Downloads", value: "downloads" },
          { label: "Form Data", value: "formData" },
          { label: "Passwords", value: "passwords" },
          { label: "IndexedDB", value: "indexedDB" },
          { label: "Local Storage", value: "localStorage" },
          { label: "Service Workers", value: "serviceWorkers" },
        ], default: ["cache", "cookies", "history", "downloads", "formData", "passwords", "indexedDB", "localStorage", "serviceWorkers"] },
        { key: "range", type: "select", label: "Time range", options: ["Last hour", "Last 24 hours", "Last 7 days", "Last 4 weeks", "All time"] },
      ],
      execute: async (inputs) => {
        const dataTypes = normalizeDataTypes(inputs.types);
        const dataToRemove = buildDataToRemove(dataTypes);
        const since = getSinceFromRange(inputs.range);
        await chrome.browsingData.remove({ since }, dataToRemove);
      }
    },

    // --- Extension Management ---
    {
      id: "enable_extension_by_name",
      label: "Enable Extension by Name",
      input: "text",
      inputs: [
        { key: "name", type: "text", label: "Extension name", required: true },
      ],
      execute: async (inputs) => {
        const name = inputs.name;
        const extensions = await chrome.management.getAll();
        const target = extensions.find(e => e.name.toLowerCase().includes(name.toLowerCase()));
        if (target) {
          await chrome.management.setEnabled(target.id, true);
        } else {
          throw new Error(`Extension "${name}" not found`);
        }
      }
    },
    {
      id: "enable_all_extensions",
      label: "Enable All Extensions",
      execute: async () => {
        const extensions = await chrome.management.getAll();
        for (const ext of extensions) {
          if (!ext.enabled && ext.type === "extension") {
            try {
              await chrome.management.setEnabled(ext.id, true);
            } catch (e) {
              logger.error(`Failed to enable extension: ${ext.name}`, e);
            }
          }
        }
      }
    },
    {
      id: "toggle_extension_by_name",
      label: "Toggle Extension by Name",
      input: "text",
      inputs: [
        { key: "name", type: "text", label: "Extension name", required: true },
      ],
      execute: async (inputs) => {
        const name = inputs.name;
        const extensions = await chrome.management.getAll();
        const target = extensions.find(e => e.name.toLowerCase().includes(name.toLowerCase()));
        if (target) {
          await chrome.management.setEnabled(target.id, !target.enabled);
        } else {
          throw new Error(`Extension "${name}" not found`);
        }
      }
    },
    {
      id: "disable_extension_by_name",
      label: "Disable Extension by Name",
      input: "text",
      inputs: [
        { key: "name", type: "text", label: "Extension name", required: true },
      ],
      execute: async (inputs) => {
        const name = inputs.name;
        const extensions = await chrome.management.getAll();
        const target = extensions.find(e => e.name.toLowerCase().includes(name.toLowerCase()));
        if (target) {
          await chrome.management.setEnabled(target.id, false);
        } else {
          throw new Error(`Extension "${name}" not found`);
        }
      }
    },
    {
      id: "disable_all_other_extensions",
      label: "Disable All Other Extensions",
      execute: async () => {
        const extensions = await chrome.management.getAll();
        const selfId = chrome.runtime.id;
        for (const ext of extensions) {
          if (ext.id !== selfId && ext.enabled && ext.type === "extension") {
            try {
              await chrome.management.setEnabled(ext.id, false);
            } catch (e) {
              logger.error(`Failed to disable extension: ${ext.name}`, e, { notify: true });
            }
          }
        }
      }
    },
    {
      id: "disable_all_extensions",
      label: "Disable All Extensions",
      execute: async () => {
        const extensions = await chrome.management.getAll();
        const selfId = chrome.runtime.id;
        for (const ext of extensions) {
          if (ext.id !== selfId && ext.enabled && ext.type === "extension") {
            try {
              await chrome.management.setEnabled(ext.id, false);
            } catch (e) {
              logger.error(`Failed to disable extension: ${ext.name}`, e, { notify: true });
            }
          }
        }
        await chrome.management.setEnabled(selfId, false);
      }
    },

    // --- Content Settings ---
    {
      id: "allow_all_settings",
      label: "Allow All Settings",
      inputs: [
        { key: "types", type: "checkboxes", label: "Settings", options: GLOBAL_CONTENT_SETTING_OPTIONS, default: GLOBAL_CONTENT_SETTING_DEFAULTS },
      ],
      execute: (inputs) => applyGlobalContentSettings(inputs?.types, "allow")
    },
    {
      id: "block_all_settings",
      label: "Block All Settings",
      inputs: [
        { key: "types", type: "checkboxes", label: "Settings", options: GLOBAL_CONTENT_SETTING_OPTIONS, default: GLOBAL_CONTENT_SETTING_DEFAULTS },
      ],
      execute: (inputs) => applyGlobalContentSettings(inputs?.types, "block")
    },
    {
      id: "toggle_all_settings",
      label: "Toggle All Settings",
      inputs: [
        { key: "types", type: "checkboxes", label: "Settings", options: GLOBAL_CONTENT_SETTING_OPTIONS, default: GLOBAL_CONTENT_SETTING_DEFAULTS },
      ],
      execute: (inputs) => applyGlobalContentSettings(inputs?.types, "toggle")
    },
    {
      id: "set_cookies_session_only",
      label: "Set Cookies to Session Only",
      execute: () => setGlobalContentSetting("cookies", "session_only")
    },
    {
      id: "reset_all_content_settings",
      label: "Reset All Content Settings",
      execute: async () => {
        for (const type of GLOBAL_CONTENT_SETTING_DEFAULTS) {
          await clearGlobalContentSetting(type);
        }
      }
    }
  ],

  reading_list: [
    // --- Opening ---
    {
      id: "open_new_tab",
      label: "Open in New Tab",
      selection: "each",
      execute: (entry) => chrome.tabs.create({ url: entry.url })
    },
    {
      id: "open_background",
      label: "Open in Background",
      selection: "each",
      execute: (entry) => chrome.tabs.create({ url: entry.url, active: false })
    },
    {
      id: "open_new_window",
      label: "Open in New Window",
      selection: "each",
      execute: (entry) => chrome.windows.create({ url: entry.url })
    },
    {
      id: "open_incognito",
      label: "Open in Incognito Window",
      selection: "each",
      execute: (entry) => chrome.windows.create({ url: entry.url, incognito: true })
    },
    {
      id: "open_in_new_group",
      label: "Open in New Group",
      selection: "group",
      minItems: 1,
      inputs: [
        { key: "color", type: "select", label: "Color", options: TAB_GROUP_COLOR_OPTIONS_WITH_UNSPECIFIED },
        { key: "name", type: "text", label: "Name" },
      ],
      execute: async (entries, inputs) => {
        const tabIds = [];
        for (const entry of entries) {
          if (!entry.url) continue;
          const tab = await chrome.tabs.create({ url: entry.url, active: false });
          tabIds.push(tab.id);
        }
        if (tabIds.length === 0) return;
        const groupId = await chrome.tabs.group({ tabIds });
        const color = (inputs.color || '').toLowerCase();
        const updateProps = {};
        if (inputs.name) updateProps.title = inputs.name;
        if (color && color !== 'unspecified') updateProps.color = color;
        if (Object.keys(updateProps).length > 0) {
          await chrome.tabGroups.update(groupId, updateProps);
        }
      }
    },

    // --- Read Status ---
    {
      id: "mark_read",
      label: "Mark as Read",
      selection: "each",
      execute: (entry) => chrome.readingList.updateEntry({ url: entry.url, hasBeenRead: true })
    },
    {
      id: "mark_unread",
      label: "Mark as Unread",
      selection: "each",
      execute: (entry) => chrome.readingList.updateEntry({ url: entry.url, hasBeenRead: false })
    },
    {
      id: "toggle_read_status",
      label: "Toggle Read Status",
      selection: "each",
      execute: async (entry) => {
        const entries = await chrome.readingList.query({ url: entry.url });
        const current = entries[0];
        if (!current) return;
        await chrome.readingList.updateEntry({ url: entry.url, hasBeenRead: !current.hasBeenRead });
      }
    },

    // --- Export / Clipboard ---
    {
      id: "copy_url_to_clipboard",
      label: "Copy URL to Clipboard",
      selection: "each",
      execute: async (entry) => {
        await copyToClipboard(entry.url);
      }
    },

    // --- Save / Convert ---
    {
      id: "add_to_bookmarks",
      label: "Add to Bookmarks",
      selection: "each",
      input: "text",
      inputs: [
        { key: "destination", type: "select", label: "Destination", options: ["(Default)", "Bookmarks Bar", "Other Bookmarks", "Custom"] },
        { key: "folder", type: "text", label: "Folder", required: true, showWhen: { key: "destination", value: "Custom" } },
      ],
      execute: async (entry, inputs) => {
        let parentId;
        if (inputs.destination && inputs.destination !== '(Default)') {
          parentId = await resolveBookmarkFolder(inputs);
        }
        await chrome.bookmarks.create({
          parentId,
          title: entry.title || entry.url,
          url: entry.url
        });
      }
    },

    // --- Removal ---
    {
      id: "remove",
      label: "Remove",
      selection: "each",
      execute: (entry) => chrome.readingList.removeEntry({ url: entry.url })
    }
  ],

  tab: [
    // --- Lifecycle ---
    {
      id: "create_new_tab",
      label: "Create New Tab",
      selection: "none",
      input: "text",
      inputs: [
        { key: "position", type: "select", label: "Position", options: ["Right of Tab", "Left of Tab", "First", "Last", "Random"] },
        { key: "url", type: "text", label: "URL" },
      ],
      execute: async (inputs) => {
        const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
        const url = inputs.url || undefined;
        const position = inputs.position || "Right of Tab";
        let index;
        if (position === "Right of Tab") {
          index = activeTab ? activeTab.index + 1 : undefined;
        } else if (position === "Left of Tab") {
          index = activeTab ? activeTab.index : undefined;
        } else if (position === "First") {
          index = 0;
        } else if (position === "Last") {
          index = undefined; // Chrome places at end by default
        } else if (position === "Random") {
          const tabs = await chrome.tabs.query({ currentWindow: true });
          index = Math.floor(Math.random() * (tabs.length + 1));
        }
        return chrome.tabs.create({ url, index });
      }
    },
    {
      id: "close",
      label: "Close",
      selection: "each",
      execute: (tab) => chrome.tabs.remove(tab.id)
    },
    {
      id: "duplicate",
      label: "Duplicate",
      selection: "each",
      execute: (tab) => chrome.tabs.duplicate(tab.id)
    },
    {
      id: "discard_tab",
      label: "Discard (Memory Saver)",
      selection: "each",
      execute: (tab) => chrome.tabs.discard(tab.id)
    },

    // --- Navigation ---
    {
      id: "reload",
      label: "Reload",
      selection: "each",
      execute: (tab) => chrome.tabs.reload(tab.id)
    },
    {
      id: "hard_reload",
      label: "Hard Reload",
      selection: "each",
      execute: (tab) => chrome.tabs.reload(tab.id, { bypassCache: true })
    },
    {
      id: "back",
      label: "Back",
      selection: "each",
      execute: (tab) => chrome.tabs.goBack(tab.id)
    },
    {
      id: "forward",
      label: "Forward",
      selection: "each",
      execute: (tab) => chrome.tabs.goForward(tab.id)
    },

    // --- Focus / Activation ---
    {
      id: "activate",
      label: "Activate",
      selection: "single",
      execute: (tab) => chrome.tabs.update(tab.id, { active: true })
    },
    {
      id: "go_to_next_tab",
      label: "Go to Next Tab",
      selection: "group",
      execute: async (tabs) => {
        if (!Array.isArray(tabs) || tabs.length === 0) return;
        const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!activeTab) return;
        const sortedTabs = [...tabs].sort((a, b) => a.index - b.index);
        const currentIndex = sortedTabs.findIndex(t => t.id === activeTab.id);
        const nextIndex = currentIndex === -1 ? 0 : (currentIndex + 1) % sortedTabs.length;
        const nextTab = sortedTabs[nextIndex];
        if (!nextTab) return;
        await chrome.tabs.update(nextTab.id, { active: true });
        await chrome.windows.update(nextTab.windowId, { focused: true });
      }
    },
    {
      id: "go_to_previous_tab",
      label: "Go to Previous Tab",
      selection: "group",
      execute: async (tabs) => {
        if (!Array.isArray(tabs) || tabs.length === 0) return;
        const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!activeTab) return;
        const sortedTabs = [...tabs].sort((a, b) => a.index - b.index);
        const currentIndex = sortedTabs.findIndex(t => t.id === activeTab.id);
        const prevIndex = currentIndex === -1 ? sortedTabs.length - 1 : (currentIndex - 1 + sortedTabs.length) % sortedTabs.length;
        const prevTab = sortedTabs[prevIndex];
        if (!prevTab) return;
        await chrome.tabs.update(prevTab.id, { active: true });
        await chrome.windows.update(prevTab.windowId, { focused: true });
      }
    },
    {
      id: "highlight",
      label: "Highlight",
      selection: "group",
      minItems: 1,
      execute: async (tabs) => {
        const tabsByWindow = new Map();
        for (const tab of tabs || []) {
          if (!tab) continue;
          if (typeof tab.windowId !== "number" || typeof tab.index !== "number") continue;
          if (!tabsByWindow.has(tab.windowId)) {
            tabsByWindow.set(tab.windowId, []);
          }
          tabsByWindow.get(tab.windowId).push(tab.index);
        }
        for (const [windowId, indices] of tabsByWindow) {
          if (indices.length > 0) {
            await chrome.tabs.highlight({ windowId, tabs: indices });
          }
        }
      }
    },

    // --- Pin ---
    {
      id: "pin",
      label: "Pin",
      selection: "each",
      execute: (tab) => chrome.tabs.update(tab.id, { pinned: true })
    },
    {
      id: "unpin",
      label: "Unpin",
      selection: "each",
      execute: (tab) => chrome.tabs.update(tab.id, { pinned: false })
    },
    {
      id: "toggle_pin",
      label: "Toggle Pin",
      selection: "each",
      execute: async (tab) => chrome.tabs.update(tab.id, { pinned: !tab.pinned })
    },

    // --- Mute ---
    {
      id: "mute",
      label: "Mute",
      selection: "each",
      execute: (tab) => chrome.tabs.update(tab.id, { muted: true })
    },
    {
      id: "unmute",
      label: "Unmute",
      selection: "each",
      execute: (tab) => chrome.tabs.update(tab.id, { muted: false })
    },
    {
      id: "toggle_mute",
      label: "Toggle Mute",
      selection: "each",
      execute: async (tab) => chrome.tabs.update(tab.id, { muted: !tab.mutedInfo?.muted })
    },

    // --- Movement (within window) ---
    {
      id: "move_left",
      label: "Move Left",
      selection: "each",
      execute: async (tab) => {
        if (tab.index > 0) {
          await chrome.tabs.move(tab.id, { index: tab.index - 1 });
        }
      }
    },
    {
      id: "move_right",
      label: "Move Right",
      selection: "each",
      execute: async (tab) => chrome.tabs.move(tab.id, { index: tab.index + 1 })
    },
    {
      id: "move_to_start",
      label: "Move to Start",
      selection: "each",
      execute: (tab) => chrome.tabs.move(tab.id, { index: 0 })
    },
    {
      id: "move_to_end",
      label: "Move to End",
      selection: "each",
      execute: (tab) => chrome.tabs.move(tab.id, { index: -1 })
    },
    {
      id: "move_to_position",
      label: "Move to Position (Index)",
      selection: "each",
      input: "text",
      inputs: [
        { key: "index", type: "number", label: "Index", required: true, step: 1, default: 0 },
      ],
      execute: async (tab, inputs) => {
        let index = parseInt(inputs.index, 10);
        if (index < 0) {
          const tabs = await chrome.tabs.query({ windowId: tab.windowId });
          index = Math.max(0, tabs.length + index);
        }
        return chrome.tabs.move(tab.id, { index });
      }
    },

    // --- Window Operations ---
    {
      id: "move_to_new_window",
      label: "Move to New Window",
      selection: "group",
      minItems: 1,
      execute: async (tabs) => {
        const tabIds = (tabs || []).map((t) => t?.id).filter((id) => typeof id === "number");
        const win = await chrome.windows.create({ tabId: tabIds[0] });
        if (tabIds.length > 1) {
          await chrome.tabs.move(tabIds.slice(1), { windowId: win.id, index: -1 });
        }
      }
    },
    {
      id: "copy_to_new_window",
      label: "Copy to New Window",
      selection: "group",
      minItems: 1,
      execute: async (tabs) => {
        const win = await chrome.windows.create({});
        const defaultTab = win.tabs && win.tabs[0];
        for (const tab of tabs || []) {
          if (!tab || !tab.url) continue;
          await chrome.tabs.create({ windowId: win.id, url: tab.url });
        }
        if (defaultTab) {
          await chrome.tabs.remove(defaultTab.id);
        }
      }
    },

    // --- Tab Groups ---
    {
      id: "add_to_new_group",
      label: "Add to New Group",
      selection: "group",
      minItems: 1,
      input: "text",
      inputs: [
        { key: "color", type: "select", label: "Color", options: TAB_GROUP_COLOR_OPTIONS_WITH_UNSPECIFIED },
        { key: "name", type: "text", label: "Name" },
      ],
      execute: async (tabs, inputs) => {
        const tabIds = (tabs || []).map((t) => t?.id).filter((id) => typeof id === "number");
        const groupId = await chrome.tabs.group({ tabIds });
        const color = inputs.color.toLowerCase();
        const updateProps = {};
        if (inputs.name) updateProps.title = inputs.name;
        if (color && color !== "unspecified") updateProps.color = color;
        if (Object.keys(updateProps).length > 0) {
          await chrome.tabGroups.update(groupId, updateProps);
        }
      }
    },
    {
      id: "add_to_existing_group",
      label: "Add to Existing Group",
      selection: "group",
      minItems: 1,
      input: "text",
      inputs: [
        { key: "groupName", type: "text", label: "Group Name", required: true },
      ],
      execute: async (tabs, inputs) => {
        const name = (inputs.groupName || '').trim();
        const groups = await chrome.tabGroups.query({});
        const targetGroup = groups.find((group) => String(group?.title || '').trim().toLowerCase() === name.toLowerCase());
        const groupId = targetGroup ? targetGroup.id : null;
        if (groupId === null) throw new Error(`Group "${name}" not found`);
        const tabIds = (tabs || []).map((t) => t?.id).filter((id) => typeof id === "number");
        await chrome.tabs.group({ tabIds, groupId });
      }
    },
    {
      id: "auto_group_by_domain",
      label: "Auto-Group by Domain",
      selection: "group",
      input: "text",
      inputs: [
        { key: "color", type: "select", label: "Color", options: TAB_GROUP_COLOR_OPTIONS_WITH_UNSPECIFIED },
      ],
      execute: async (tabs, inputs) => {
        const domainMap = new Map();
        for (const tab of tabs || []) {
          try {
            if (!tab || !tab.url) continue;
            const domain = new URL(tab.url).hostname;
            if (!domainMap.has(domain)) {
              domainMap.set(domain, []);
            }
            domainMap.get(domain).push(tab.id);
          } catch {
            // ignore invalid URLs
          }
        }

        const color = inputs.color.toLowerCase();
        for (const [domain, ids] of domainMap) {
          if (ids.length > 1) {
            const groupId = await chrome.tabs.group({ tabIds: ids });
            const updateProps = { title: domain };
            if (color && color !== "unspecified") updateProps.color = color;
            await chrome.tabGroups.update(groupId, updateProps);
          }
        }
      }
    },
    {
      id: "remove_from_group",
      label: "Remove from Group",
      selection: "each",
      execute: (tab) => chrome.tabs.ungroup(tab.id)
    },

    // --- Deduplication ---
    {
      id: "close_duplicate_tabs_keep_first",
      label: "Close Duplicate Tabs (Keep First)",
      selection: "group",
      execute: async (tabs) => {
        const seen = new Set();
        const toClose = [];
        for (const tab of tabs) {
          if (!tab.url) continue;
          if (seen.has(tab.url)) {
            toClose.push(tab.id);
          } else {
            seen.add(tab.url);
          }
        }
        if (toClose.length > 0) {
          await chrome.tabs.remove(toClose);
        }
      }
    },
    {
      id: "close_duplicate_tabs_keep_last",
      label: "Close Duplicate Tabs (Keep Last)",
      selection: "group",
      execute: async (tabs) => {
        const lastByUrl = new Map();
        for (const tab of tabs) {
          if (!tab.url) continue;
          lastByUrl.set(tab.url, tab.id);
        }
        const keepIds = new Set(lastByUrl.values());
        const toClose = [];
        for (const tab of tabs) {
          if (tab.url && !keepIds.has(tab.id)) {
            toClose.push(tab.id);
          }
        }
        if (toClose.length > 0) {
          await chrome.tabs.remove(toClose);
        }
      }
    },

    // --- Sorting / Ordering ---
    {
      id: "sort_by_url",
      label: "Sort by URL",
      selection: "group",
      execute: async (tabs) => {
        const sorted = [...tabs].sort((a, b) => (a.url || "").localeCompare(b.url || ""));
        for (const [i, tab] of sorted.entries()) {
          await chrome.tabs.move(tab.id, { index: i });
        }
      }
    },
    {
      id: "sort_by_title",
      label: "Sort by Title",
      selection: "group",
      execute: async (tabs) => {
        const sorted = [...tabs].sort((a, b) => (a.title || "").localeCompare(b.title || ""));
        for (const [i, tab] of sorted.entries()) {
          await chrome.tabs.move(tab.id, { index: i });
        }
      }
    },
    {
      id: "shuffle",
      label: "Shuffle",
      selection: "group",
      execute: async (tabs) => {
        const shuffled = shuffleArray(tabs);
        for (const [i, tab] of shuffled.entries()) {
          await chrome.tabs.move(tab.id, { index: i });
        }
      }
    },

    // --- Zoom ---
    {
      id: "zoom_in",
      label: "Zoom In",
      selection: "each",
      input: "text",
      inputs: [
        { key: "step", type: "text", label: "Step", default: "0.1" },
      ],
      execute: async (tab, inputs) => {
        const zoom = await chrome.tabs.getZoom(tab.id);
        const value = String(inputs.step ?? '').trim();
        let delta;
        if (!value) {
          delta = 0.1;
        } else if (value.endsWith('%')) {
          const percent = parseFloat(value.slice(0, -1));
          delta = percent / 100;
        } else {
          const numeric = parseFloat(value);
          delta = numeric <= 1 ? numeric : numeric / 100;
        }
        await chrome.tabs.setZoom(tab.id, Math.min(zoom + delta, ZOOM_LIMITS.max));
      }
    },
    {
      id: "zoom_out",
      label: "Zoom Out",
      selection: "each",
      input: "text",
      inputs: [
        { key: "step", type: "text", label: "Step", default: "0.1" },
      ],
      execute: async (tab, inputs) => {
        const zoom = await chrome.tabs.getZoom(tab.id);
        const value = String(inputs.step ?? '').trim();
        let delta;
        if (!value) {
          delta = 0.1;
        } else if (value.endsWith('%')) {
          const percent = parseFloat(value.slice(0, -1));
          delta = percent / 100;
        } else {
          const numeric = parseFloat(value);
          delta = numeric <= 1 ? numeric : numeric / 100;
        }
        await chrome.tabs.setZoom(tab.id, Math.max(zoom - delta, ZOOM_LIMITS.min));
      }
    },
    {
      id: "set_zoom",
      label: "Set Zoom Level",
      selection: "each",
      input: "text",
      inputs: [
        { key: "level", type: "number", label: "Zoom level", default: 1.0, required: true, min: ZOOM_LIMITS.min, max: ZOOM_LIMITS.max, step: 0.05 },
      ],
      execute: (tab, inputs) => {
        const level = parseFloat(inputs.level);
        if (!isNaN(level) && level >= ZOOM_LIMITS.min && level <= ZOOM_LIMITS.max) {
          return chrome.tabs.setZoom(tab.id, level);
        }
      }
    },

    // --- Export / Clipboard ---
    {
      id: "copy_urls_to_clipboard",
      label: "Copy URLs to Clipboard",
      selection: "group",
      minItems: 1,
      execute: async (tabs) => {
        const text = (tabs || []).map(t => t.url).filter(Boolean).join("\n");
        await copyToClipboard(text);
      }
    },
    {
      id: "download_tab_inventory_json",
      label: "Download Tab Inventory (JSON)",
      selection: "group",
      minItems: 1,
      execute: async (tabs) => {
        const json = JSON.stringify(tabs || [], null, 2);
        const blob = new Blob([json], { type: "application/json" });
        const url = URL.createObjectURL(blob);
        await chrome.downloads.download({ url, filename: "tab-inventory.json" });
      }
    },

    // --- Screenshots ---
    {
      id: "screenshot_to_downloads",
      label: "Screenshot to Downloads",
      selection: "each",
      inputs: [
        { key: "format", type: "select", label: "Format", options: ["PNG", "JPG"] },
        { key: "filename", type: "text", label: "Filename", default: "{{url}}-{{date}}" },
      ],
      execute: async (tab, inputs) => {
        try {
          await chrome.tabs.update(tab.id, { active: true });
          await new Promise(r => setTimeout(r, 100));
          const formatMap = { 'PNG': 'png', 'JPG': 'jpeg' };
          const extMap = { 'PNG': '.png', 'JPG': '.jpg' };
          const chosen = inputs.format || 'PNG';
          const format = formatMap[chosen] || 'png';
          const ext = extMap[chosen] || '.png';
          const dataUrl = await chrome.tabs.captureVisibleTab(tab.windowId, { format });

          const dateStr = new Date().toISOString().split('T')[0];
          let filename = (inputs.filename || '{{url}}-{{date}}')
            .replace(/\{\{date\}\}/g, dateStr)
            .replace(/\{\{title\}\}/g, tab.title || String(tab.id))
            .replace(/\{\{url\}\}/g, safeHostname(tab.url));
          filename = filename.replace(/[<>:"/\\|?*]+/g, '_').replace(/_{2,}/g, '_').replace(/^_|_$/g, '') + ext;
          await chrome.downloads.download({ url: dataUrl, filename });
        } catch (e) {
          logger.error("Screenshot failed:", e, { notify: true });
        }
      }
    },
    {
      id: "screenshot_to_clipboard",
      label: "Screenshot to Clipboard",
      selection: "single",
      execute: async (tab) => {
        try {
          await chrome.tabs.update(tab.id, { active: true });
          await new Promise(r => setTimeout(r, 100));
          const dataUrl = await chrome.tabs.captureVisibleTab(tab.windowId, { format: "png" });
          await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            func: async (url) => {
              const res = await fetch(url);
              const blob = await res.blob();
              const item = new ClipboardItem({ [blob.type]: blob });
              await navigator.clipboard.write([item]);
            },
            args: [dataUrl]
          });
        } catch (e) {
          logger.error("Screenshot to clipboard failed:", e, { notify: true });
        }
      }
    },

    // --- URL Manipulation ---
    {
      id: "find_replace_in_url",
      label: "Find & Replace in URL",
      selection: "each",
      input: "text",
      inputs: [
        { key: "find", type: "text", label: "Find (Regex)", required: true },
        { key: "replace", type: "text", label: "Replace" },
      ],
      execute: async (tab, inputs) => {
        const replace = inputs.replace || "";
        const regex = new RegExp(inputs.find, "g");
        const newUrl = tab.url.replace(regex, replace);
        if (newUrl !== tab.url) {
          await chrome.tabs.update(tab.id, { url: newUrl });
        }
      }
    },

    // --- Page Actions ---
    {
      id: "print",
      label: "Print",
      selection: "each",
      execute: async (tab) => {
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: () => window.print()
        });
      }
    },
    {
      id: "scroll_to_top",
      label: "Scroll to Top",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
      ],
      execute: async (tab, inputs) => {
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (scrollBehavior) => {
            window.scrollTo({ top: 0, behavior: scrollBehavior });
          },
          args: [behavior]
        });
      }
    },
    {
      id: "scroll_to_bottom",
      label: "Scroll to Bottom",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
      ],
      execute: async (tab, inputs) => {
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (scrollBehavior) => {
            const maxScrollTop = Math.max(
              document.body?.scrollHeight || 0,
              document.documentElement?.scrollHeight || 0
            );
            window.scrollTo({ top: maxScrollTop, behavior: scrollBehavior });
          },
          args: [behavior]
        });
      }
    },
    {
      id: "scroll_up",
      label: "Scroll Up",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
        { key: "amount", type: "number", label: "Pixels", default: 300, min: 1, step: 1 },
      ],
      execute: async (tab, inputs) => {
        const parsedAmount = parseInt(inputs?.amount, 10);
        const amount = !isNaN(parsedAmount) && parsedAmount > 0 ? parsedAmount : 300;
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (pixels, scrollBehavior) => {
            window.scrollBy({ top: -pixels, left: 0, behavior: scrollBehavior });
          },
          args: [amount, behavior]
        });
      }
    },
    {
      id: "scroll_down",
      label: "Scroll Down",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
        { key: "amount", type: "number", label: "Pixels", default: 300, min: 1, step: 1 },
      ],
      execute: async (tab, inputs) => {
        const parsedAmount = parseInt(inputs?.amount, 10);
        const amount = !isNaN(parsedAmount) && parsedAmount > 0 ? parsedAmount : 300;
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (pixels, scrollBehavior) => {
            window.scrollBy({ top: pixels, left: 0, behavior: scrollBehavior });
          },
          args: [amount, behavior]
        });
      }
    },
    {
      id: "page_up",
      label: "Page Up",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
      ],
      execute: async (tab, inputs) => {
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (scrollBehavior) => {
            window.scrollBy({ top: -window.innerHeight, left: 0, behavior: scrollBehavior });
          },
          args: [behavior]
        });
      }
    },
    {
      id: "page_down",
      label: "Page Down",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
      ],
      execute: async (tab, inputs) => {
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (scrollBehavior) => {
            window.scrollBy({ top: window.innerHeight, left: 0, behavior: scrollBehavior });
          },
          args: [behavior]
        });
      }
    },
    {
      id: "scroll_left",
      label: "Scroll Left",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
        { key: "amount", type: "number", label: "Pixels", default: 300, min: 1, step: 1 },
      ],
      execute: async (tab, inputs) => {
        const parsedAmount = parseInt(inputs?.amount, 10);
        const amount = !isNaN(parsedAmount) && parsedAmount > 0 ? parsedAmount : 300;
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (pixels, scrollBehavior) => {
            window.scrollBy({ top: 0, left: -pixels, behavior: scrollBehavior });
          },
          args: [amount, behavior]
        });
      }
    },
    {
      id: "scroll_right",
      label: "Scroll Right",
      selection: "each",
      inputs: [
        { key: "smooth", type: "boolean", label: "Smooth Scroll", default: false },
        { key: "amount", type: "number", label: "Pixels", default: 300, min: 1, step: 1 },
      ],
      execute: async (tab, inputs) => {
        const parsedAmount = parseInt(inputs?.amount, 10);
        const amount = !isNaN(parsedAmount) && parsedAmount > 0 ? parsedAmount : 300;
        const behavior = inputs?.smooth === true ? "smooth" : "auto";
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (pixels, scrollBehavior) => {
            window.scrollBy({ top: 0, left: pixels, behavior: scrollBehavior });
          },
          args: [amount, behavior]
        });
      }
    },
    {
      id: "share_page",
      label: "Share",
      selection: "each",
      execute: async (tab) => {
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: async () => {
            if (typeof navigator.share !== "function") {
              throw new Error("Web Share API is not available on this page");
            }
            await navigator.share({ title: document.title, url: location.href });
          }
        });
      }
    },
    {
      id: "enable_edit_mode",
      label: "Enable Page Edit Mode",
      selection: "each",
      execute: async (tab) => {
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: () => {
            document.designMode = "on";
          }
        });
      }
    },
    {
      id: "disable_edit_mode",
      label: "Disable Page Edit Mode",
      selection: "each",
      execute: async (tab) => {
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: () => {
            document.designMode = "off";
          }
        });
      }
    },
    {
      id: "toggle_edit_mode",
      label: "Toggle Page Edit Mode",
      selection: "each",
      execute: async (tab) => {
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: () => {
            document.designMode = document.designMode === "on" ? "off" : "on";
          }
        });
      }
    },

    // --- Scripting ---
    {
      id: "run_javascript",
      label: "Run JavaScript",
      selection: "each",
      input: "textarea",
      inputs: [
        { key: "code", type: "textarea", label: "JavaScript", required: true },
      ],
      execute: async (tab, inputs) => {
        const code = String(inputs.code || "");
        await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          func: (userCode) => {
            (0, eval)(userCode);
          },
          args: [code]
        });
      }
    },
    {
      id: "insert_css",
      label: "Insert CSS",
      selection: "each",
      input: "textarea",
      inputs: [
        { key: "css", type: "textarea", label: "CSS", required: true },
      ],
      execute: async (tab, inputs) => {
        const css = String(inputs.css || "");
        await chrome.scripting.insertCSS({
          target: { tabId: tab.id },
          css
        });
      }
    },
    {
      id: "remove_css",
      label: "Remove Inserted CSS",
      selection: "each",
      input: "textarea",
      inputs: [
        { key: "css", type: "textarea", label: "CSS", required: true },
      ],
      execute: async (tab, inputs) => {
        const css = String(inputs.css || "");
        await chrome.scripting.removeCSS({
          target: { tabId: tab.id },
          css
        });
      }
    },

    // --- Save / Convert ---
    {
      id: "add_to_bookmarks",
      label: "Add to Bookmarks",
      selection: "each",
      input: "text",
      inputs: [
        { key: "destination", type: "select", label: "Destination", options: ["(Default)", "Bookmarks Bar", "Other Bookmarks", "Custom"] },
        { key: "folder", type: "text", label: "Folder", required: true, showWhen: { key: "destination", value: "Custom" } },
      ],
      execute: async (tab, inputs) => {
        let parentId;
        if (inputs.destination && inputs.destination !== '(Default)') {
          parentId = await resolveBookmarkFolder(inputs);
        }
        await chrome.bookmarks.create({
          parentId,
          title: tab.title || tab.url,
          url: tab.url
        });
      }
    },
    {
      id: "add_to_reading_list",
      label: "Add to Reading List",
      selection: "each",
      execute: async (tab) => {
        await chrome.readingList.addEntry({
          url: tab.url,
          title: tab.title,
          hasBeenRead: false
        });
      }
    },

    // --- Session Restore ---
    {
      id: "restore_session",
      label: "Restore Session",
      selection: "each",
      execute: async (tab) => {
        if (!tab.sessionId) return;
        await chrome.sessions.restore(tab.sessionId);
      }
    },

    // --- Site Settings ---
    {
      id: "allow_site_settings",
      label: "Allow Site Settings",
      selection: "each",
      inputs: [
        { key: "types", type: "checkboxes", label: "Settings", options: SITE_CONTENT_SETTING_OPTIONS, default: SITE_CONTENT_SETTING_DEFAULTS },
      ],
      execute: (tab, inputs) => applySiteContentSettings(tab, inputs?.types, "allow")
    },
    {
      id: "block_site_settings",
      label: "Block Site Settings",
      selection: "each",
      inputs: [
        { key: "types", type: "checkboxes", label: "Settings", options: SITE_CONTENT_SETTING_OPTIONS, default: SITE_CONTENT_SETTING_DEFAULTS },
      ],
      execute: (tab, inputs) => applySiteContentSettings(tab, inputs?.types, "block")
    },
    {
      id: "toggle_site_settings",
      label: "Toggle Site Settings",
      selection: "each",
      inputs: [
        { key: "types", type: "checkboxes", label: "Settings", options: SITE_CONTENT_SETTING_OPTIONS, default: SITE_CONTENT_SETTING_DEFAULTS },
      ],
      execute: (tab, inputs) => applySiteContentSettings(tab, inputs?.types, "toggle")
    },
    {
      id: "clear_browsing_data_this_domain",
      label: "Clear Browsing Data (This Domain)",
      selection: "each",
      inputs: [
        { key: "types", type: "checkboxes", label: "Data types", options: [
          { label: "Cache", value: "cache" },
          { label: "Cookies", value: "cookies" },
          { label: "History", value: "history" },
          { label: "Downloads", value: "downloads" },
          { label: "Form Data", value: "formData" },
          { label: "Passwords", value: "passwords" },
          { label: "IndexedDB", value: "indexedDB" },
          { label: "Local Storage", value: "localStorage" },
          { label: "Service Workers", value: "serviceWorkers" },
        ], default: ["cache", "cookies", "history", "downloads", "formData", "passwords", "indexedDB", "localStorage", "serviceWorkers"] },
        { key: "range", type: "select", label: "Time range", options: ["Last hour", "Last 24 hours", "Last 7 days", "Last 4 weeks", "All time"] },
      ],
      execute: async (tab, inputs) => {
        const dataTypes = normalizeDataTypes(inputs.types);
        const dataToRemove = buildDataToRemove(dataTypes);
        const since = getSinceFromRange(inputs.range);
        if (!tab?.url) return;
        const url = new URL(tab.url);
        if (url.protocol !== "http:" && url.protocol !== "https:") return;
        await chrome.browsingData.remove({ since, origins: [url.origin] }, dataToRemove);
      }
    }
  ],

  window: [
    // --- Tab Creation ---
    {
      id: "create_new_tab",
      label: "Create New Tab",
      input: "text",
      inputs: [
        { key: "position", type: "select", label: "Position", options: ["First", "Last", "Random"] },
        { key: "url", type: "text", label: "URL" },
      ],
      execute: async (win, inputs) => {
        const windowId = (typeof win === "number" ? win : win?.id);
        const url = inputs.url || undefined;
        const position = inputs.position || "First";
        let index;
        if (position === "First") {
          index = 0;
        } else if (position === "Last") {
          index = undefined; // Chrome places at end by default
        } else if (position === "Random") {
          const currentWindow = await chrome.windows.get(windowId, { populate: true });
          const tabCount = currentWindow.tabs ? currentWindow.tabs.length : 0;
          index = Math.floor(Math.random() * (tabCount + 1));
        }
        await chrome.tabs.create({ windowId, url, index });
      }
    },

    // --- Focus ---
    {
      id: "focus",
      label: "Focus",
      excludeSelections: ["Focused Window"],
      execute: (win) => chrome.windows.update((typeof win === "number" ? win : win?.id), { focused: true })
    },
    {
      id: "go_to_next_window",
      label: "Go to Next Window",
      selection: "group",
      execute: async (windows) => {
        if (!Array.isArray(windows) || windows.length === 0) return;
        const focusedWindow = await chrome.windows.getLastFocused();
        if (!focusedWindow) return;
        const sortedWindows = [...windows].sort((a, b) => a.id - b.id);
        const currentIndex = sortedWindows.findIndex(w => w.id === focusedWindow.id);
        const nextIndex = currentIndex === -1 ? 0 : (currentIndex + 1) % sortedWindows.length;
        const nextWindow = sortedWindows[nextIndex];
        if (!nextWindow) return;
        await chrome.windows.update(nextWindow.id, { focused: true });
      }
    },
    {
      id: "go_to_previous_window",
      label: "Go to Previous Window",
      selection: "group",
      execute: async (windows) => {
        if (!Array.isArray(windows) || windows.length === 0) return;
        const focusedWindow = await chrome.windows.getLastFocused();
        if (!focusedWindow) return;
        const sortedWindows = [...windows].sort((a, b) => a.id - b.id);
        const currentIndex = sortedWindows.findIndex(w => w.id === focusedWindow.id);
        const prevIndex = currentIndex === -1 ? sortedWindows.length - 1 : (currentIndex - 1 + sortedWindows.length) % sortedWindows.length;
        const prevWindow = sortedWindows[prevIndex];
        if (!prevWindow) return;
        await chrome.windows.update(prevWindow.id, { focused: true });
      }
    },

    // --- State ---
    {
      id: "minimize",
      label: "Minimize",
      execute: (win) => chrome.windows.update((typeof win === "number" ? win : win?.id), { state: "minimized" })
    },
    {
      id: "maximize",
      label: "Maximize",
      execute: (win) => chrome.windows.update((typeof win === "number" ? win : win?.id), { state: "maximized" })
    },
    {
      id: "fullscreen",
      label: "Fullscreen",
      execute: (win) => chrome.windows.update((typeof win === "number" ? win : win?.id), { state: "fullscreen" })
    },
    {
      id: "restore_normal",
      label: "Restore (Normal)",
      execute: (win) => chrome.windows.update((typeof win === "number" ? win : win?.id), { state: "normal" })
    },
    {
      id: "toggle_maximize",
      label: "Toggle Maximize",
      execute: async (win) => {
        const windowId = (typeof win === "number" ? win : win?.id);
        const current = await chrome.windows.get(windowId);
        const nextState = current.state === "maximized" ? "normal" : "maximized";
        await chrome.windows.update(windowId, { state: nextState });
      }
    },
    {
      id: "toggle_fullscreen",
      label: "Toggle Fullscreen",
      execute: async (win) => {
        const windowId = (typeof win === "number" ? win : win?.id);
        const current = await chrome.windows.get(windowId);
        const nextState = current.state === "fullscreen" ? "normal" : "fullscreen";
        await chrome.windows.update(windowId, { state: nextState });
      }
    },
    {
      id: "draw_attention",
      label: "Draw Attention",
      execute: (win) => chrome.windows.update((typeof win === "number" ? win : win?.id), { drawAttention: true })
    },

    // --- Size & Position ---
    {
      id: "resize",
      label: "Resize",
      input: "text",
      inputs: [
        { key: "width", type: "number", label: "Width" },
        { key: "height", type: "number", label: "Height" },
      ],
      execute: (win, inputs) => {
        const windowId = (typeof win === "number" ? win : win?.id);
        const width = parseInt(inputs.width);
        const height = parseInt(inputs.height);
        const update = {};
        if (!isNaN(width)) update.width = width;
        if (!isNaN(height)) update.height = height;
        if (Object.keys(update).length > 0) {
          return chrome.windows.update(windowId, update);
        }
      }
    },
    {
      id: "move_position",
      label: "Move Position",
      input: "text",
      inputs: [
        { key: "left", type: "number", label: "Left" },
        { key: "top", type: "number", label: "Top" },
      ],
      execute: (win, inputs) => {
        const windowId = (typeof win === "number" ? win : win?.id);
        const left = parseInt(inputs.left);
        const top = parseInt(inputs.top);
        const update = {};
        if (!isNaN(left)) update.left = left;
        if (!isNaN(top)) update.top = top;
        if (Object.keys(update).length > 0) {
          return chrome.windows.update(windowId, update);
        }
      }
    },

    // --- Duplication ---
    {
      id: "duplicate",
      label: "Duplicate",
      execute: async (win) => {
        const windowId = (typeof win === "number" ? win : win?.id);
        const currentWindow = await chrome.windows.get(windowId, { populate: true });
        const newWin = await chrome.windows.create({});
        const defaultTab = newWin.tabs && newWin.tabs[0];
        for (const tab of currentWindow.tabs || []) {
          if (!tab || !tab.url) continue;
          await chrome.tabs.create({ windowId: newWin.id, url: tab.url });
        }
        if (defaultTab) {
          await chrome.tabs.remove(defaultTab.id);
        }
      }
    },

    // --- Tab Management ---
    {
      id: "merge_tabs",
      label: "Merge Tabs into Focused Window",
      execute: async (win) => {
        const windowId = (typeof win === "number" ? win : win?.id);
        const focusedWindow = await chrome.windows.getLastFocused();
        if (windowId !== focusedWindow.id) {
          const currentWindow = await chrome.windows.get(windowId, { populate: true });
          if (currentWindow.tabs && currentWindow.tabs.length > 0) {
            const tabIds = currentWindow.tabs.map(t => t.id);
            await chrome.tabs.move(tabIds, { windowId: focusedWindow.id, index: -1 });
          }
        }
      }
    },

    // --- Session Restore ---
    {
      id: "restore_session",
      label: "Restore Session",
      execute: async (win) => {
        if (win.sessionId) {
          await chrome.sessions.restore(win.sessionId);
        }
      }
    },

    // --- Close ---
    {
      id: "close",
      label: "Close",
      execute: (win) => chrome.windows.remove((typeof win === "number" ? win : win?.id))
    }
  ],

  group: [
    // --- Collapse State ---
    {
      id: "expand",
      label: "Expand",
      selection: "each",
      execute: (group) => chrome.tabGroups.update((typeof group === "number" ? group : group?.id), { collapsed: false })
    },
    {
      id: "collapse",
      label: "Collapse",
      selection: "each",
      execute: (group) => chrome.tabGroups.update((typeof group === "number" ? group : group?.id), { collapsed: true })
    },
    {
      id: "toggle_collapse",
      label: "Toggle Collapse",
      selection: "each",
      execute: async (group) => {
        const groupId = (typeof group === "number" ? group : group?.id);
        const currentGroup = await chrome.tabGroups.get(groupId);
        await chrome.tabGroups.update(groupId, { collapsed: !currentGroup.collapsed });
      }
    },

    // --- Appearance ---
    {
      id: "set_title",
      label: "Set Title",
      selection: "each",
      input: "text",
      inputs: [
        { key: "name", type: "text", label: "Title", required: true },
      ],
      execute: (group, inputs) => chrome.tabGroups.update((typeof group === "number" ? group : group?.id), { title: inputs.name })
    },
    {
      id: "set_color",
      label: "Set Color",
      selection: "each",
      input: "select",
      inputs: [
        { key: "color", type: "select", label: "Color", options: TAB_GROUP_COLOR_OPTIONS },
      ],
      execute: (group, inputs) => chrome.tabGroups.update((typeof group === "number" ? group : group?.id), { color: inputs.color })
    },

    // --- Window Operations ---
    {
      id: "move_to_new_window",
      label: "Move to New Window",
      selection: "each",
      execute: async (group) => {
        const groupId = (typeof group === "number" ? group : group?.id);
        // Get group properties before moving (group will be destroyed when tabs leave)
        const currentGroup = await chrome.tabGroups.get(groupId);
        const { title, color } = currentGroup;

        const tabs = await chrome.tabs.query({ groupId });
        if (tabs.length === 0) return;
        const tabIds = tabs.map(t => t.id);
        const win = await chrome.windows.create({ tabId: tabIds[0] });
        if (tabIds.length > 1) {
          await chrome.tabs.move(tabIds.slice(1), { windowId: win.id, index: -1 });
        }
        // Create a new group in the new window with the same properties
        const newGroupId = await chrome.tabs.group({ tabIds, createProperties: { windowId: win.id } });
        await chrome.tabGroups.update(newGroupId, { title, color });
      }
    },

    // --- Dissolution ---
    {
      id: "close_all_tabs",
      label: "Close All Tabs",
      selection: "each",
      execute: async (group) => {
        const groupId = (typeof group === "number" ? group : group?.id);
        const tabs = await chrome.tabs.query({ groupId });
        const tabIds = tabs.map(t => t.id);
        if (tabIds.length === 0) return;
        await chrome.tabs.remove(tabIds);
      }
    },
    {
      id: "ungroup",
      label: "Ungroup",
      selection: "each",
      execute: async (group) => {
        const groupId = (typeof group === "number" ? group : group?.id);
        const tabs = await chrome.tabs.query({ groupId });
        const tabIds = tabs.map(t => t.id);
        if (tabIds.length === 0) return;
        await chrome.tabs.ungroup(tabIds);
      }
    }
  ]
};

function getContentSettingApi(type) {
  const api = chrome.contentSettings?.[type];
  if (!api) {
    throw new Error(`Unsupported content setting type: ${type}`);
  }
  return api;
}

async function setGlobalContentSetting(type, setting) {
  const api = getContentSettingApi(type);
  await api.set({
    primaryPattern: "<all_urls>",
    setting
  });
}

async function toggleGlobalContentSetting(type) {
  const api = getContentSettingApi(type);
  const current = await api.get({ primaryUrl: "http://example.com/" });
  const nextSetting = current.setting === "allow" ? "block" : "allow";
  await api.set({
    primaryPattern: "<all_urls>",
    setting: nextSetting
  });
}

async function clearGlobalContentSetting(type) {
  const api = getContentSettingApi(type);
  await api.clear({});
}

function getSitePattern(tab) {
  const rawUrl = String(tab?.url || "");
  if (!rawUrl) return null;
  const parsed = new URL(rawUrl);
  if (parsed.protocol !== "http:" && parsed.protocol !== "https:") {
    return null;
  }
  return `${parsed.protocol}//${parsed.hostname}/*`;
}

function normalizeContentSettingTypes(types, defaults) {
  if (Array.isArray(types) && types.length > 0) {
    return types;
  }
  return defaults;
}

function normalizeGlobalContentSettingTypes(types) {
  return normalizeContentSettingTypes(types, GLOBAL_CONTENT_SETTING_DEFAULTS);
}

function normalizeSiteContentSettingTypes(types) {
  return normalizeContentSettingTypes(types, SITE_CONTENT_SETTING_DEFAULTS);
}

async function setSiteContentSetting(type, tab, setting) {
  const pattern = getSitePattern(tab);
  if (!pattern) return;
  const api = getContentSettingApi(type);
  await api.set({
    primaryPattern: pattern,
    setting
  });
}

async function toggleSiteContentSetting(type, tab) {
  const pattern = getSitePattern(tab);
  if (!pattern || !tab?.url) return;
  const api = getContentSettingApi(type);
  const current = await api.get({ primaryUrl: tab.url });
  const nextSetting = current.setting === "allow" ? "block" : "allow";
  await api.set({
    primaryPattern: pattern,
    setting: nextSetting
  });
}

async function applySiteContentSettings(tab, types, mode) {
  const normalizedTypes = normalizeSiteContentSettingTypes(types);
  for (const type of normalizedTypes) {
    if (mode === "toggle") {
      await toggleSiteContentSetting(type, tab);
    } else {
      await setSiteContentSetting(type, tab, mode);
    }
  }
}

async function applyGlobalContentSettings(types, mode) {
  const normalizedTypes = normalizeGlobalContentSettingTypes(types);
  for (const type of normalizedTypes) {
    if (mode === "toggle") {
      await toggleGlobalContentSetting(type);
    } else {
      await setGlobalContentSetting(type, mode);
    }
  }
}

function normalizeDataTypes(types) {
  if (Array.isArray(types) && types.length > 0) {
    return types;
  }
  return ["cache", "cookies", "history", "downloads", "formData", "passwords", "indexedDB", "localStorage", "serviceWorkers"];
}

function buildDataToRemove(types) {
  return types.reduce((acc, type) => {
    acc[type] = true;
    return acc;
  }, {});
}

function getSinceFromRange(range) {
  const now = Date.now();
  const ranges = {
    "Last hour": 60 * 60 * 1000,
    "Last 24 hours": 24 * 60 * 60 * 1000,
    "Last 7 days": 7 * 24 * 60 * 60 * 1000,
    "Last 4 weeks": 28 * 24 * 60 * 60 * 1000
  };
  const offset = ranges[range] || 0;
  return offset === 0 ? 0 : now - offset;
}

async function getSystemInfo(types = null) {
  const results = {};
  const checks = {
    cpu: () => chrome.system.cpu.getInfo(),
    memory: () => chrome.system.memory.getInfo(),
    storage: () => chrome.system.storage.getInfo(),
    display: () => chrome.system.display.getInfo(),
    extensions: () => chrome.management.getAll(),
    topSites: () => chrome.topSites.get(),
    fonts: () => chrome.fontSettings.getFontList(),
    tabs: () => chrome.tabs.query({})
  };

  const keysToFetch = types || Object.keys(checks);
  for (const key of keysToFetch) {
    const fetcher = checks[key];
    if (!fetcher) continue;
    try {
      results[key] = await fetcher();
    } catch (e) {
      results[key] = { error: e.message };
    }
  }
  return results;
}

async function copyToClipboard(text, mode = "Plain text") {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  if (!tab?.id) throw new Error("No active tab available for clipboard operation");
  await chrome.scripting.executeScript({
    target: { tabId: tab.id },
    func: async (payload) => {
      const value = String(payload?.text || "");
      const clipboardMode = payload?.mode || "Plain text";

      try {
        if (clipboardMode === "HTML" && navigator.clipboard?.write && typeof ClipboardItem !== "undefined") {
          const plainTextContainer = document.createElement("div");
          plainTextContainer.innerHTML = value;
          const plainText = plainTextContainer.textContent || value;

          await navigator.clipboard.write([
            new ClipboardItem({
              "text/html": new Blob([value], { type: "text/html" }),
              "text/plain": new Blob([plainText], { type: "text/plain" }),
            })
          ]);
          return;
        }

        await navigator.clipboard.writeText(value);
      } catch {
        // Fallback when clipboard API fails (e.g. no user gesture in confirm mode)
        const ta = document.createElement('textarea');
        ta.value = value;
        ta.style.position = 'fixed';
        ta.style.opacity = '0';
        document.body.appendChild(ta);
        ta.select();
        document.execCommand('copy');
        ta.remove();
      }
    },
    args: [{ text, mode }]
  });
}

async function resolveBookmarkFolderByPath(folderPath, { createIfMissing = false } = {}) {
  const normalizedPath = String(folderPath || '').trim().replace(/\\/g, '/').replace(/\/{2,}/g, '/').replace(/^\/+|\/+$/g, '');
  if (!normalizedPath) return null;

  const ROOT_PATH_MAP = {
    'bookmarks bar': '1',
    'other bookmarks': '2',
    'mobile bookmarks': '3',
  };

  // First, try to find a folder whose title is literally the full path string (with slashes)
  const results = await chrome.bookmarks.search({ title: normalizedPath });
  const literalMatch = results.find(b => !b.url && b.title === normalizedPath);
  if (literalMatch) return literalMatch.id;

  // If not found and path contains slashes, walk the tree.
  // Optionally create missing segments while walking.
  if (normalizedPath.includes('/')) {
    const segments = normalizedPath.split('/').filter(Boolean);
    if (segments.length === 0) return null;

    const rootId = ROOT_PATH_MAP[segments[0].toLowerCase()];
    let parentId = rootId || '2';
    let startIndex = 0;

    if (rootId) {
      startIndex = 1;
      if (segments.length === 1) {
        return rootId;
      }
    }

    let candidates = await chrome.bookmarks.getChildren(parentId);
    for (const [index, segment] of segments.entries()) {
      if (index < startIndex) continue;
      let match = findFolderInChildren(candidates, segment);
      if (!match) {
        if (!createIfMissing) return null;
        match = await chrome.bookmarks.create({ title: segment, parentId });
      }
      parentId = match.id;
      candidates = await chrome.bookmarks.getChildren(parentId);
    }
    return parentId;
  }

  // Simple name lookup
  const folder = results.find(b => !b.url);
  if (folder) return folder.id;
  if (!createIfMissing) return null;

  const created = await chrome.bookmarks.create({ title: normalizedPath, parentId: '2' });
  return created.id;
}

function findFolderInChildren(children, name) {
  for (const child of children) {
    if (!child.url && child.title.toLowerCase() === name.toLowerCase()) {
      return child;
    }
  }
  return null;
}

async function resolveBookmarkFolder(inputs) {
  const dest = inputs.destination || 'Custom';
  if (dest === 'Bookmarks Bar') return '1';
  if (dest === 'Other Bookmarks') return '2';

  const folderPath = (inputs.folder || '').trim();
  if (!folderPath) {
    throw new Error('Custom folder path is required');
  }

  const id = await resolveBookmarkFolderByPath(folderPath, { createIfMissing: true });
  if (!id) {
    throw new Error(`Folder "${folderPath}" not found`);
  }
  return id;
}

async function resolveBookmarkParentFolder(inputs) {
  const dest = inputs.destination || 'Bookmarks Bar';
  if (dest === 'Bookmarks Bar') return '1';
  if (dest === 'Other Bookmarks') return '2';

  const parentFolder = (inputs.parentFolder || '').trim();
  if (!parentFolder) {
    throw new Error('Custom parent folder name is required');
  }

  const id = await resolveBookmarkFolderByPath(parentFolder, { createIfMissing: true });
  if (!id) {
    throw new Error(`Folder "${parentFolder}" not found`);
  }
  return id;
}


